//! A simple color struct with constants. Defines `Rgb` and `Rgba`

use std::borrow::Cow;

use crate::utils::{Vector3, Vector4};

macro_rules! def_color
{
    ($name:ident $($field:ident)*) => 
    {
        #[derive(Debug, Clone, Copy, PartialEq, serde::Serialize)]
        pub struct $name
        {
            $(pub $field: u8),*
        }
        impl $name
        {
            pub fn new($($field: u8),*) -> Self
            {
                Self { $($field),* }
            }
        }
        impl std::fmt::Display for $name
        {
            fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result 
            {
                write!(f, "#")?;
                $(write!(f, "{:02x}", self.$field)?;)*
                Ok(())
            }
        }
        impl From<$name> for Cow<'_, str>
        {
            fn from(color: $name) -> Self 
            {
                Cow::Owned(color.to_string())
            }
        }
    }
}

def_color!(Rgb r g b);
def_color!(Rgba r g b a);

impl From<&Rgb> for Vector3
{
    fn from(color: &Rgb) -> Self 
    {
        Self 
        { 
            x: color.r as f64 / 255.0, 
            y: color.g as f64 / 255.0, 
            z: color.b as f64 / 255.0 
        }
    }
}

impl From<&Rgba> for Vector4
{
    fn from(color: &Rgba) -> Self 
    {
        Self 
        { 
            x: color.r as f64 / 255.0, 
            y: color.g as f64 / 255.0, 
            z: color.b as f64 / 255.0, 
            w: color.a as f64 / 255.0 
        }
    }
}

impl From<&Rgb> for Rgba
{
    fn from(color: &Rgb) -> Self 
    {
        Self 
        { 
            r: color.r, 
            g: color.g, 
            b: color.b,
            a: 255
        }
    }
}

impl From<&Rgba> for Rgb
{
    fn from(color: &Rgba) -> Self 
    {
        Self 
        { 
            r: color.r, 
            g: color.g, 
            b: color.b
        }
    }
}

/// This is a simple transparency.
pub const TRANSPARENT: Rgba = Rgba { r: 0, g: 0, b: 0, a: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFFFFF;margin: 10px 10px 10px 10px"></div>
pub const WHITE: Rgb = Rgb { r: 255, g: 255, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#000000;margin: 10px 10px 10px 10px"></div>
pub const BLACK: Rgb = Rgb { r: 0, g: 0, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFFAFA;margin: 10px 10px 10px 10px"></div>
pub const SNOW: Rgb = Rgb { r: 255, g: 250, b: 250 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F8F8FF;margin: 10px 10px 10px 10px"></div>
pub const GHOSTWHITE: Rgb = Rgb { r: 249, g: 249, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F5F5F5;margin: 10px 10px 10px 10px"></div>
pub const WHITESMOKE: Rgb = Rgb { r: 246, g: 246, b: 246 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#DCDCDC;margin: 10px 10px 10px 10px"></div>
pub const GAINSBORO: Rgb = Rgb { r: 220, g: 220, b: 220 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFFAF0;margin: 10px 10px 10px 10px"></div>
pub const FLORALWHITE: Rgb = Rgb { r: 255, g: 250, b: 240 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FDF5E6;margin: 10px 10px 10px 10px"></div>
pub const OLDLACE: Rgb = Rgb { r: 253, g: 246, b: 230 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FAF0E6;margin: 10px 10px 10px 10px"></div>
pub const LINEN: Rgb = Rgb { r: 250, g: 240, b: 230 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FAEBD7;margin: 10px 10px 10px 10px"></div>
pub const ANTIQUEWHITE: Rgb = Rgb { r: 250, g: 236, b: 215 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFEFD5;margin: 10px 10px 10px 10px"></div>
pub const PAPAYAWHIP: Rgb = Rgb { r: 255, g: 239, b: 213 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFEBCD;margin: 10px 10px 10px 10px"></div>
pub const BLANCHEDALMOND: Rgb = Rgb { r: 255, g: 236, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFE4C4;margin: 10px 10px 10px 10px"></div>
pub const BISQUE: Rgb = Rgb { r: 255, g: 228, b: 196 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFDAB9;margin: 10px 10px 10px 10px"></div>
pub const PEACHPUFF: Rgb = Rgb { r: 255, g: 218, b: 185 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFDEAD;margin: 10px 10px 10px 10px"></div>
pub const NAVAJOWHITE: Rgb = Rgb { r: 255, g: 222, b: 173 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFE4B5;margin: 10px 10px 10px 10px"></div>
pub const MOCCASIN: Rgb = Rgb { r: 255, g: 228, b: 181 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFF8DC;margin: 10px 10px 10px 10px"></div>
pub const CORNSILK: Rgb = Rgb { r: 255, g: 249, b: 220 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFFFF0;margin: 10px 10px 10px 10px"></div>
pub const IVORY: Rgb = Rgb { r: 255, g: 255, b: 240 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFFACD;margin: 10px 10px 10px 10px"></div>
pub const LEMONCHIFFON: Rgb = Rgb { r: 255, g: 250, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFF5EE;margin: 10px 10px 10px 10px"></div>
pub const SEASHELL: Rgb = Rgb { r: 255, g: 246, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F0FFF0;margin: 10px 10px 10px 10px"></div>
pub const HONEYDEW: Rgb = Rgb { r: 240, g: 255, b: 240 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F5FFFA;margin: 10px 10px 10px 10px"></div>
pub const MINTCREAM: Rgb = Rgb { r: 246, g: 255, b: 250 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F0FFFF;margin: 10px 10px 10px 10px"></div>
pub const AZURE: Rgb = Rgb { r: 240, g: 255, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F0F8FF;margin: 10px 10px 10px 10px"></div>
pub const ALICEBLUE: Rgb = Rgb { r: 240, g: 249, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#E6E6FA;margin: 10px 10px 10px 10px"></div>
pub const LAVENDER: Rgb = Rgb { r: 230, g: 230, b: 250 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFF0F5;margin: 10px 10px 10px 10px"></div>
pub const LAVENDERBLUSH: Rgb = Rgb { r: 255, g: 240, b: 246 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFE4E1;margin: 10px 10px 10px 10px"></div>
pub const MISTYROSE: Rgb = Rgb { r: 255, g: 228, b: 225 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#2F4F4F;margin: 10px 10px 10px 10px"></div>
pub const DARKSLATEGREY: Rgb = Rgb { r: 47, g: 79, b: 79 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#696969;margin: 10px 10px 10px 10px"></div>
pub const DIMGREY: Rgb = Rgb { r: 105, g: 105, b: 105 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#708090;margin: 10px 10px 10px 10px"></div>
pub const SLATEGREY: Rgb = Rgb { r: 112, g: 128, b: 144 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#778899;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSLATEGREY: Rgb = Rgb { r: 119, g: 136, b: 153 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#BEBEBE;margin: 10px 10px 10px 10px"></div>
pub const GREY: Rgb = Rgb { r: 190, g: 190, b: 190 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#D3D3D3;margin: 10px 10px 10px 10px"></div>
pub const LIGHTGREY: Rgb = Rgb { r: 211, g: 211, b: 211 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#191970;margin: 10px 10px 10px 10px"></div>
pub const MIDNIGHTBLUE: Rgb = Rgb { r: 25, g: 25, b: 112 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#000080;margin: 10px 10px 10px 10px"></div>
pub const NAVY: Rgb = Rgb { r: 0, g: 0, b: 128 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#000080;margin: 10px 10px 10px 10px"></div>
pub const NAVYBLUE: Rgb = Rgb { r: 0, g: 0, b: 128 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#6495ED;margin: 10px 10px 10px 10px"></div>
pub const CORNFLOWERBLUE: Rgb = Rgb { r: 100, g: 149, b: 237 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#483D8B;margin: 10px 10px 10px 10px"></div>
pub const DARKSLATEBLUE: Rgb = Rgb { r: 72, g: 61, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#6A5ACD;margin: 10px 10px 10px 10px"></div>
pub const SLATEBLUE: Rgb = Rgb { r: 106, g: 90, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7B68EE;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMSLATEBLUE: Rgb = Rgb { r: 123, g: 104, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8470FF;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSLATEBLUE: Rgb = Rgb { r: 132, g: 112, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#0000CD;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMBLUE: Rgb = Rgb { r: 0, g: 0, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#4169E1;margin: 10px 10px 10px 10px"></div>
pub const ROYALBLUE: Rgb = Rgb { r: 65, g: 105, b: 225 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#0000FF;margin: 10px 10px 10px 10px"></div>
pub const BLUE: Rgb = Rgb { r: 0, g: 0, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#1E90FF;margin: 10px 10px 10px 10px"></div>
pub const DODGERBLUE: Rgb = Rgb { r: 30, g: 144, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00BFFF;margin: 10px 10px 10px 10px"></div>
pub const DEEPSKYBLUE: Rgb = Rgb { r: 0, g: 191, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#87CEEB;margin: 10px 10px 10px 10px"></div>
pub const SKYBLUE: Rgb = Rgb { r: 135, g: 206, b: 236 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#87CEFA;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSKYBLUE: Rgb = Rgb { r: 135, g: 206, b: 250 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#4682B4;margin: 10px 10px 10px 10px"></div>
pub const STEELBLUE: Rgb = Rgb { r: 70, g: 130, b: 180 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B0C4DE;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSTEELBLUE: Rgb = Rgb { r: 176, g: 196, b: 222 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#ADD8E6;margin: 10px 10px 10px 10px"></div>
pub const LIGHTBLUE: Rgb = Rgb { r: 173, g: 216, b: 230 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B0E0E6;margin: 10px 10px 10px 10px"></div>
pub const POWDERBLUE: Rgb = Rgb { r: 176, g: 224, b: 230 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#AFEEEE;margin: 10px 10px 10px 10px"></div>
pub const PALETURQUOISE: Rgb = Rgb { r: 175, g: 238, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00CED1;margin: 10px 10px 10px 10px"></div>
pub const DARKTURQUOISE: Rgb = Rgb { r: 0, g: 206, b: 209 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#48D1CC;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMTURQUOISE: Rgb = Rgb { r: 72, g: 209, b: 204 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#40E0D0;margin: 10px 10px 10px 10px"></div>
pub const TURQUOISE: Rgb = Rgb { r: 64, g: 224, b: 208 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00FFFF;margin: 10px 10px 10px 10px"></div>
pub const CYAN: Rgb = Rgb { r: 0, g: 255, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#E0FFFF;margin: 10px 10px 10px 10px"></div>
pub const LIGHTCYAN: Rgb = Rgb { r: 224, g: 255, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#5F9EA0;margin: 10px 10px 10px 10px"></div>
pub const CADETBLUE: Rgb = Rgb { r: 95, g: 158, b: 160 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#66CDAA;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMAQUAMARINE: Rgb = Rgb { r: 102, g: 205, b: 170 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7FFFD4;margin: 10px 10px 10px 10px"></div>
pub const AQUAMARINE: Rgb = Rgb { r: 127, g: 255, b: 212 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#006400;margin: 10px 10px 10px 10px"></div>
pub const DARKGREEN: Rgb = Rgb { r: 0, g: 100, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#556B2F;margin: 10px 10px 10px 10px"></div>
pub const DARKOLIVEGREEN: Rgb = Rgb { r: 85, g: 107, b: 47 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8FBC8F;margin: 10px 10px 10px 10px"></div>
pub const DARKSEAGREEN: Rgb = Rgb { r: 143, g: 188, b: 143 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#2E8B57;margin: 10px 10px 10px 10px"></div>
pub const SEAGREEN: Rgb = Rgb { r: 46, g: 139, b: 87 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#3CB371;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMSEAGREEN: Rgb = Rgb { r: 60, g: 179, b: 113 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#20B2AA;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSEAGREEN: Rgb = Rgb { r: 32, g: 178, b: 170 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#98FB98;margin: 10px 10px 10px 10px"></div>
pub const PALEGREEN: Rgb = Rgb { r: 152, g: 251, b: 152 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00FF7F;margin: 10px 10px 10px 10px"></div>
pub const SPRINGGREEN: Rgb = Rgb { r: 0, g: 255, b: 127 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7CFC00;margin: 10px 10px 10px 10px"></div>
pub const LAWNGREEN: Rgb = Rgb { r: 124, g: 252, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00FF00;margin: 10px 10px 10px 10px"></div>
pub const GREEN: Rgb = Rgb { r: 0, g: 255, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7FFF00;margin: 10px 10px 10px 10px"></div>
pub const CHARTREUSE: Rgb = Rgb { r: 127, g: 255, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00FA9A;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMSPRINGGREEN: Rgb = Rgb { r: 0, g: 250, b: 154 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#ADFF2F;margin: 10px 10px 10px 10px"></div>
pub const GREENYELLOW: Rgb = Rgb { r: 173, g: 255, b: 47 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#32CD32;margin: 10px 10px 10px 10px"></div>
pub const LIMEGREEN: Rgb = Rgb { r: 50, g: 205, b: 50 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#9ACD32;margin: 10px 10px 10px 10px"></div>
pub const YELLOWGREEN: Rgb = Rgb { r: 154, g: 205, b: 50 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#228B22;margin: 10px 10px 10px 10px"></div>
pub const FORESTGREEN: Rgb = Rgb { r: 34, g: 139, b: 34 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#6B8E23;margin: 10px 10px 10px 10px"></div>
pub const OLIVEDRAB: Rgb = Rgb { r: 107, g: 142, b: 35 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#BDB76B;margin: 10px 10px 10px 10px"></div>
pub const DARKKHAKI: Rgb = Rgb { r: 189, g: 183, b: 107 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F0E68C;margin: 10px 10px 10px 10px"></div>
pub const KHAKI: Rgb = Rgb { r: 240, g: 230, b: 140 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEE8AA;margin: 10px 10px 10px 10px"></div>
pub const PALEGOLDENROD: Rgb = Rgb { r: 238, g: 232, b: 170 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FAFAD2;margin: 10px 10px 10px 10px"></div>
pub const LIGHTGOLDENRODYELLOW: Rgb = Rgb { r: 250, g: 250, b: 210 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFFFE0;margin: 10px 10px 10px 10px"></div>
pub const LIGHTYELLOW: Rgb = Rgb { r: 255, g: 255, b: 224 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFFF00;margin: 10px 10px 10px 10px"></div>
pub const YELLOW: Rgb = Rgb { r: 255, g: 255, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFD700;margin: 10px 10px 10px 10px"></div>
pub const GOLD: Rgb = Rgb { r: 255, g: 215, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEDD82;margin: 10px 10px 10px 10px"></div>
pub const LIGHTGOLDENROD: Rgb = Rgb { r: 238, g: 221, b: 130 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#DAA520;margin: 10px 10px 10px 10px"></div>
pub const GOLDENROD: Rgb = Rgb { r: 218, g: 165, b: 32 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B8860B;margin: 10px 10px 10px 10px"></div>
pub const DARKGOLDENROD: Rgb = Rgb { r: 184, g: 134, b: 11 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#BC8F8F;margin: 10px 10px 10px 10px"></div>
pub const ROSYBROWN: Rgb = Rgb { r: 188, g: 143, b: 143 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD5C5C;margin: 10px 10px 10px 10px"></div>
pub const INDIANRED: Rgb = Rgb { r: 205, g: 92, b: 92 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B4513;margin: 10px 10px 10px 10px"></div>
pub const SADDLEBROWN: Rgb = Rgb { r: 139, g: 69, b: 19 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#A0522D;margin: 10px 10px 10px 10px"></div>
pub const SIENNA: Rgb = Rgb { r: 160, g: 82, b: 45 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD853F;margin: 10px 10px 10px 10px"></div>
pub const PERU: Rgb = Rgb { r: 205, g: 133, b: 63 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#DEB887;margin: 10px 10px 10px 10px"></div>
pub const BURLYWOOD: Rgb = Rgb { r: 222, g: 184, b: 135 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F5F5DC;margin: 10px 10px 10px 10px"></div>
pub const BEIGE: Rgb = Rgb { r: 246, g: 246, b: 220 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F5DEB3;margin: 10px 10px 10px 10px"></div>
pub const WHEAT: Rgb = Rgb { r: 246, g: 222, b: 179 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F4A460;margin: 10px 10px 10px 10px"></div>
pub const SANDYBROWN: Rgb = Rgb { r: 244, g: 164, b: 96 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#D2B48C;margin: 10px 10px 10px 10px"></div>
pub const TAN: Rgb = Rgb { r: 210, g: 180, b: 140 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#D2691E;margin: 10px 10px 10px 10px"></div>
pub const CHOCOLATE: Rgb = Rgb { r: 210, g: 105, b: 30 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B22222;margin: 10px 10px 10px 10px"></div>
pub const FIREBRICK: Rgb = Rgb { r: 178, g: 34, b: 34 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#A52A2A;margin: 10px 10px 10px 10px"></div>
pub const BROWN: Rgb = Rgb { r: 165, g: 42, b: 42 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#E9967A;margin: 10px 10px 10px 10px"></div>
pub const DARKSALMON: Rgb = Rgb { r: 233, g: 150, b: 122 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FA8072;margin: 10px 10px 10px 10px"></div>
pub const SALMON: Rgb = Rgb { r: 250, g: 128, b: 114 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFA07A;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSALMON: Rgb = Rgb { r: 255, g: 160, b: 122 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFA500;margin: 10px 10px 10px 10px"></div>
pub const ORANGE: Rgb = Rgb { r: 255, g: 165, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF8C00;margin: 10px 10px 10px 10px"></div>
pub const DARKORANGE: Rgb = Rgb { r: 255, g: 140, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF7F50;margin: 10px 10px 10px 10px"></div>
pub const CORAL: Rgb = Rgb { r: 255, g: 127, b: 80 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F08080;margin: 10px 10px 10px 10px"></div>
pub const LIGHTCORAL: Rgb = Rgb { r: 240, g: 128, b: 128 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF6347;margin: 10px 10px 10px 10px"></div>
pub const TOMATO: Rgb = Rgb { r: 255, g: 99, b: 71 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF4500;margin: 10px 10px 10px 10px"></div>
pub const ORANGERED: Rgb = Rgb { r: 255, g: 69, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF0000;margin: 10px 10px 10px 10px"></div>
pub const RED: Rgb = Rgb { r: 255, g: 0, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF69B4;margin: 10px 10px 10px 10px"></div>
pub const HOTPINK: Rgb = Rgb { r: 255, g: 105, b: 180 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF1493;margin: 10px 10px 10px 10px"></div>
pub const DEEPPINK: Rgb = Rgb { r: 255, g: 19, b: 147 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFC0CB;margin: 10px 10px 10px 10px"></div>
pub const PINK: Rgb = Rgb { r: 255, g: 192, b: 203 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFB6C1;margin: 10px 10px 10px 10px"></div>
pub const LIGHTPINK: Rgb = Rgb { r: 255, g: 182, b: 193 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#DB7093;margin: 10px 10px 10px 10px"></div>
pub const PALEVIOLETRED: Rgb = Rgb { r: 219, g: 112, b: 147 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B03060;margin: 10px 10px 10px 10px"></div>
pub const MAROON: Rgb = Rgb { r: 176, g: 48, b: 96 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#C71585;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMVIOLETRED: Rgb = Rgb { r: 199, g: 20, b: 133 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#D02090;margin: 10px 10px 10px 10px"></div>
pub const VIOLETRED: Rgb = Rgb { r: 208, g: 32, b: 144 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF00FF;margin: 10px 10px 10px 10px"></div>
pub const MAGENTA: Rgb = Rgb { r: 255, g: 0, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE82EE;margin: 10px 10px 10px 10px"></div>
pub const VIOLET: Rgb = Rgb { r: 238, g: 130, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#DDA0DD;margin: 10px 10px 10px 10px"></div>
pub const PLUM: Rgb = Rgb { r: 221, g: 160, b: 221 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#DA70D6;margin: 10px 10px 10px 10px"></div>
pub const ORCHID: Rgb = Rgb { r: 218, g: 112, b: 214 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#BA55D3;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMORCHID: Rgb = Rgb { r: 186, g: 85, b: 211 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#9932CC;margin: 10px 10px 10px 10px"></div>
pub const DARKORCHID: Rgb = Rgb { r: 153, g: 50, b: 204 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#9400D3;margin: 10px 10px 10px 10px"></div>
pub const DARKVIOLET: Rgb = Rgb { r: 148, g: 0, b: 211 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8A2BE2;margin: 10px 10px 10px 10px"></div>
pub const BLUEVIOLET: Rgb = Rgb { r: 138, g: 43, b: 226 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#A020F0;margin: 10px 10px 10px 10px"></div>
pub const PURPLE: Rgb = Rgb { r: 160, g: 32, b: 240 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#9370DB;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMPURPLE: Rgb = Rgb { r: 147, g: 112, b: 219 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#D8BFD8;margin: 10px 10px 10px 10px"></div>
pub const THISTLE: Rgb = Rgb { r: 216, g: 191, b: 216 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFFAFA;margin: 10px 10px 10px 10px"></div>
pub const SNOW1: Rgb = Rgb { r: 255, g: 250, b: 250 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEE9E9;margin: 10px 10px 10px 10px"></div>
pub const SNOW2: Rgb = Rgb { r: 238, g: 233, b: 233 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDC9C9;margin: 10px 10px 10px 10px"></div>
pub const SNOW3: Rgb = Rgb { r: 205, g: 201, b: 201 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B8989;margin: 10px 10px 10px 10px"></div>
pub const SNOW4: Rgb = Rgb { r: 139, g: 137, b: 137 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFF5EE;margin: 10px 10px 10px 10px"></div>
pub const SEASHELL1: Rgb = Rgb { r: 255, g: 246, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEE5DE;margin: 10px 10px 10px 10px"></div>
pub const SEASHELL2: Rgb = Rgb { r: 238, g: 229, b: 222 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDC5BF;margin: 10px 10px 10px 10px"></div>
pub const SEASHELL3: Rgb = Rgb { r: 205, g: 197, b: 191 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B8682;margin: 10px 10px 10px 10px"></div>
pub const SEASHELL4: Rgb = Rgb { r: 139, g: 134, b: 130 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFEFDB;margin: 10px 10px 10px 10px"></div>
pub const ANTIQUEWHITE1: Rgb = Rgb { r: 255, g: 239, b: 219 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEDFCC;margin: 10px 10px 10px 10px"></div>
pub const ANTIQUEWHITE2: Rgb = Rgb { r: 238, g: 224, b: 204 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDC0B0;margin: 10px 10px 10px 10px"></div>
pub const ANTIQUEWHITE3: Rgb = Rgb { r: 205, g: 192, b: 176 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B8378;margin: 10px 10px 10px 10px"></div>
pub const ANTIQUEWHITE4: Rgb = Rgb { r: 139, g: 131, b: 120 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFE4C4;margin: 10px 10px 10px 10px"></div>
pub const BISQUE1: Rgb = Rgb { r: 255, g: 228, b: 196 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EED5B7;margin: 10px 10px 10px 10px"></div>
pub const BISQUE2: Rgb = Rgb { r: 238, g: 213, b: 183 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDB79E;margin: 10px 10px 10px 10px"></div>
pub const BISQUE3: Rgb = Rgb { r: 205, g: 183, b: 158 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B7D6B;margin: 10px 10px 10px 10px"></div>
pub const BISQUE4: Rgb = Rgb { r: 139, g: 125, b: 107 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFDAB9;margin: 10px 10px 10px 10px"></div>
pub const PEACHPUFF1: Rgb = Rgb { r: 255, g: 218, b: 185 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EECBAD;margin: 10px 10px 10px 10px"></div>
pub const PEACHPUFF2: Rgb = Rgb { r: 238, g: 203, b: 173 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDAF95;margin: 10px 10px 10px 10px"></div>
pub const PEACHPUFF3: Rgb = Rgb { r: 205, g: 175, b: 149 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B7765;margin: 10px 10px 10px 10px"></div>
pub const PEACHPUFF4: Rgb = Rgb { r: 139, g: 119, b: 101 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFDEAD;margin: 10px 10px 10px 10px"></div>
pub const NAVAJOWHITE1: Rgb = Rgb { r: 255, g: 222, b: 173 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EECFA1;margin: 10px 10px 10px 10px"></div>
pub const NAVAJOWHITE2: Rgb = Rgb { r: 238, g: 207, b: 161 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDB38B;margin: 10px 10px 10px 10px"></div>
pub const NAVAJOWHITE3: Rgb = Rgb { r: 205, g: 179, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B795E;margin: 10px 10px 10px 10px"></div>
pub const NAVAJOWHITE4: Rgb = Rgb { r: 139, g: 121, b: 94 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFFACD;margin: 10px 10px 10px 10px"></div>
pub const LEMONCHIFFON1: Rgb = Rgb { r: 255, g: 250, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEE9BF;margin: 10px 10px 10px 10px"></div>
pub const LEMONCHIFFON2: Rgb = Rgb { r: 238, g: 233, b: 191 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDC9A5;margin: 10px 10px 10px 10px"></div>
pub const LEMONCHIFFON3: Rgb = Rgb { r: 205, g: 201, b: 165 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B8970;margin: 10px 10px 10px 10px"></div>
pub const LEMONCHIFFON4: Rgb = Rgb { r: 139, g: 137, b: 112 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFF8DC;margin: 10px 10px 10px 10px"></div>
pub const CORNSILK1: Rgb = Rgb { r: 255, g: 249, b: 220 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEE8CD;margin: 10px 10px 10px 10px"></div>
pub const CORNSILK2: Rgb = Rgb { r: 238, g: 232, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDC8B1;margin: 10px 10px 10px 10px"></div>
pub const CORNSILK3: Rgb = Rgb { r: 205, g: 200, b: 177 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B8878;margin: 10px 10px 10px 10px"></div>
pub const CORNSILK4: Rgb = Rgb { r: 139, g: 136, b: 120 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFFFF0;margin: 10px 10px 10px 10px"></div>
pub const IVORY1: Rgb = Rgb { r: 255, g: 255, b: 240 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEEEE0;margin: 10px 10px 10px 10px"></div>
pub const IVORY2: Rgb = Rgb { r: 238, g: 238, b: 224 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDCDC1;margin: 10px 10px 10px 10px"></div>
pub const IVORY3: Rgb = Rgb { r: 205, g: 205, b: 193 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B8B83;margin: 10px 10px 10px 10px"></div>
pub const IVORY4: Rgb = Rgb { r: 139, g: 139, b: 131 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F0FFF0;margin: 10px 10px 10px 10px"></div>
pub const HONEYDEW1: Rgb = Rgb { r: 240, g: 255, b: 240 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#E0EEE0;margin: 10px 10px 10px 10px"></div>
pub const HONEYDEW2: Rgb = Rgb { r: 224, g: 238, b: 224 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#C1CDC1;margin: 10px 10px 10px 10px"></div>
pub const HONEYDEW3: Rgb = Rgb { r: 193, g: 205, b: 193 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#838B83;margin: 10px 10px 10px 10px"></div>
pub const HONEYDEW4: Rgb = Rgb { r: 131, g: 139, b: 131 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFF0F5;margin: 10px 10px 10px 10px"></div>
pub const LAVENDERBLUSH1: Rgb = Rgb { r: 255, g: 240, b: 246 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEE0E5;margin: 10px 10px 10px 10px"></div>
pub const LAVENDERBLUSH2: Rgb = Rgb { r: 238, g: 224, b: 229 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDC1C5;margin: 10px 10px 10px 10px"></div>
pub const LAVENDERBLUSH3: Rgb = Rgb { r: 205, g: 193, b: 197 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B8386;margin: 10px 10px 10px 10px"></div>
pub const LAVENDERBLUSH4: Rgb = Rgb { r: 139, g: 131, b: 134 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFE4E1;margin: 10px 10px 10px 10px"></div>
pub const MISTYROSE1: Rgb = Rgb { r: 255, g: 228, b: 225 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EED5D2;margin: 10px 10px 10px 10px"></div>
pub const MISTYROSE2: Rgb = Rgb { r: 238, g: 213, b: 210 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDB7B5;margin: 10px 10px 10px 10px"></div>
pub const MISTYROSE3: Rgb = Rgb { r: 205, g: 183, b: 181 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B7D7B;margin: 10px 10px 10px 10px"></div>
pub const MISTYROSE4: Rgb = Rgb { r: 139, g: 125, b: 123 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F0FFFF;margin: 10px 10px 10px 10px"></div>
pub const AZURE1: Rgb = Rgb { r: 240, g: 255, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#E0EEEE;margin: 10px 10px 10px 10px"></div>
pub const AZURE2: Rgb = Rgb { r: 224, g: 238, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#C1CDCD;margin: 10px 10px 10px 10px"></div>
pub const AZURE3: Rgb = Rgb { r: 193, g: 205, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#838B8B;margin: 10px 10px 10px 10px"></div>
pub const AZURE4: Rgb = Rgb { r: 131, g: 139, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#836FFF;margin: 10px 10px 10px 10px"></div>
pub const SLATEBLUE1: Rgb = Rgb { r: 131, g: 111, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7A67EE;margin: 10px 10px 10px 10px"></div>
pub const SLATEBLUE2: Rgb = Rgb { r: 122, g: 103, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#6959CD;margin: 10px 10px 10px 10px"></div>
pub const SLATEBLUE3: Rgb = Rgb { r: 105, g: 89, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#473C8B;margin: 10px 10px 10px 10px"></div>
pub const SLATEBLUE4: Rgb = Rgb { r: 71, g: 60, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#4876FF;margin: 10px 10px 10px 10px"></div>
pub const ROYALBLUE1: Rgb = Rgb { r: 72, g: 118, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#436EEE;margin: 10px 10px 10px 10px"></div>
pub const ROYALBLUE2: Rgb = Rgb { r: 67, g: 110, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#3A5FCD;margin: 10px 10px 10px 10px"></div>
pub const ROYALBLUE3: Rgb = Rgb { r: 58, g: 95, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#27408B;margin: 10px 10px 10px 10px"></div>
pub const ROYALBLUE4: Rgb = Rgb { r: 39, g: 64, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#0000FF;margin: 10px 10px 10px 10px"></div>
pub const BLUE1: Rgb = Rgb { r: 0, g: 0, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#0000EE;margin: 10px 10px 10px 10px"></div>
pub const BLUE2: Rgb = Rgb { r: 0, g: 0, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#0000CD;margin: 10px 10px 10px 10px"></div>
pub const BLUE3: Rgb = Rgb { r: 0, g: 0, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00008B;margin: 10px 10px 10px 10px"></div>
pub const BLUE4: Rgb = Rgb { r: 0, g: 0, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#1E90FF;margin: 10px 10px 10px 10px"></div>
pub const DODGERBLUE1: Rgb = Rgb { r: 30, g: 144, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#1C86EE;margin: 10px 10px 10px 10px"></div>
pub const DODGERBLUE2: Rgb = Rgb { r: 28, g: 134, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#1874CD;margin: 10px 10px 10px 10px"></div>
pub const DODGERBLUE3: Rgb = Rgb { r: 24, g: 116, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#104E8B;margin: 10px 10px 10px 10px"></div>
pub const DODGERBLUE4: Rgb = Rgb { r: 16, g: 78, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#63B8FF;margin: 10px 10px 10px 10px"></div>
pub const STEELBLUE1: Rgb = Rgb { r: 99, g: 184, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#5CACEE;margin: 10px 10px 10px 10px"></div>
pub const STEELBLUE2: Rgb = Rgb { r: 92, g: 172, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#4F94CD;margin: 10px 10px 10px 10px"></div>
pub const STEELBLUE3: Rgb = Rgb { r: 79, g: 148, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#36648B;margin: 10px 10px 10px 10px"></div>
pub const STEELBLUE4: Rgb = Rgb { r: 54, g: 100, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00BFFF;margin: 10px 10px 10px 10px"></div>
pub const DEEPSKYBLUE1: Rgb = Rgb { r: 0, g: 191, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00B2EE;margin: 10px 10px 10px 10px"></div>
pub const DEEPSKYBLUE2: Rgb = Rgb { r: 0, g: 178, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#009ACD;margin: 10px 10px 10px 10px"></div>
pub const DEEPSKYBLUE3: Rgb = Rgb { r: 0, g: 154, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00688B;margin: 10px 10px 10px 10px"></div>
pub const DEEPSKYBLUE4: Rgb = Rgb { r: 0, g: 104, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#87CEFF;margin: 10px 10px 10px 10px"></div>
pub const SKYBLUE1: Rgb = Rgb { r: 135, g: 206, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7EC0EE;margin: 10px 10px 10px 10px"></div>
pub const SKYBLUE2: Rgb = Rgb { r: 126, g: 192, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#6CA6CD;margin: 10px 10px 10px 10px"></div>
pub const SKYBLUE3: Rgb = Rgb { r: 108, g: 166, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#4A708B;margin: 10px 10px 10px 10px"></div>
pub const SKYBLUE4: Rgb = Rgb { r: 74, g: 112, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B0E2FF;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSKYBLUE1: Rgb = Rgb { r: 176, g: 226, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#A4D3EE;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSKYBLUE2: Rgb = Rgb { r: 164, g: 211, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8DB6CD;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSKYBLUE3: Rgb = Rgb { r: 141, g: 182, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#607B8B;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSKYBLUE4: Rgb = Rgb { r: 96, g: 123, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CAE1FF;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSTEELBLUE1: Rgb = Rgb { r: 202, g: 225, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#BCD2EE;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSTEELBLUE2: Rgb = Rgb { r: 188, g: 210, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#A2B5CD;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSTEELBLUE3: Rgb = Rgb { r: 162, g: 181, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#6E7B8B;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSTEELBLUE4: Rgb = Rgb { r: 110, g: 123, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#BFEFFF;margin: 10px 10px 10px 10px"></div>
pub const LIGHTBLUE1: Rgb = Rgb { r: 191, g: 239, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B2DFEE;margin: 10px 10px 10px 10px"></div>
pub const LIGHTBLUE2: Rgb = Rgb { r: 178, g: 224, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#9AC0CD;margin: 10px 10px 10px 10px"></div>
pub const LIGHTBLUE3: Rgb = Rgb { r: 154, g: 192, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#68838B;margin: 10px 10px 10px 10px"></div>
pub const LIGHTBLUE4: Rgb = Rgb { r: 104, g: 131, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#E0FFFF;margin: 10px 10px 10px 10px"></div>
pub const LIGHTCYAN1: Rgb = Rgb { r: 224, g: 255, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#D1EEEE;margin: 10px 10px 10px 10px"></div>
pub const LIGHTCYAN2: Rgb = Rgb { r: 209, g: 238, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B4CDCD;margin: 10px 10px 10px 10px"></div>
pub const LIGHTCYAN3: Rgb = Rgb { r: 180, g: 205, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7A8B8B;margin: 10px 10px 10px 10px"></div>
pub const LIGHTCYAN4: Rgb = Rgb { r: 122, g: 139, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#BBFFFF;margin: 10px 10px 10px 10px"></div>
pub const PALETURQUOISE1: Rgb = Rgb { r: 187, g: 255, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#AEEEEE;margin: 10px 10px 10px 10px"></div>
pub const PALETURQUOISE2: Rgb = Rgb { r: 174, g: 238, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#96CDCD;margin: 10px 10px 10px 10px"></div>
pub const PALETURQUOISE3: Rgb = Rgb { r: 150, g: 205, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#668B8B;margin: 10px 10px 10px 10px"></div>
pub const PALETURQUOISE4: Rgb = Rgb { r: 102, g: 139, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#98F5FF;margin: 10px 10px 10px 10px"></div>
pub const CADETBLUE1: Rgb = Rgb { r: 152, g: 246, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8EE5EE;margin: 10px 10px 10px 10px"></div>
pub const CADETBLUE2: Rgb = Rgb { r: 142, g: 229, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7AC5CD;margin: 10px 10px 10px 10px"></div>
pub const CADETBLUE3: Rgb = Rgb { r: 122, g: 197, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#53868B;margin: 10px 10px 10px 10px"></div>
pub const CADETBLUE4: Rgb = Rgb { r: 83, g: 134, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00F5FF;margin: 10px 10px 10px 10px"></div>
pub const TURQUOISE1: Rgb = Rgb { r: 0, g: 246, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00E5EE;margin: 10px 10px 10px 10px"></div>
pub const TURQUOISE2: Rgb = Rgb { r: 0, g: 229, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00C5CD;margin: 10px 10px 10px 10px"></div>
pub const TURQUOISE3: Rgb = Rgb { r: 0, g: 197, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00868B;margin: 10px 10px 10px 10px"></div>
pub const TURQUOISE4: Rgb = Rgb { r: 0, g: 134, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00FFFF;margin: 10px 10px 10px 10px"></div>
pub const CYAN1: Rgb = Rgb { r: 0, g: 255, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00EEEE;margin: 10px 10px 10px 10px"></div>
pub const CYAN2: Rgb = Rgb { r: 0, g: 238, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00CDCD;margin: 10px 10px 10px 10px"></div>
pub const CYAN3: Rgb = Rgb { r: 0, g: 205, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#008B8B;margin: 10px 10px 10px 10px"></div>
pub const CYAN4: Rgb = Rgb { r: 0, g: 139, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7FFFD4;margin: 10px 10px 10px 10px"></div>
pub const AQUAMARINE1: Rgb = Rgb { r: 127, g: 255, b: 212 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#76EEC6;margin: 10px 10px 10px 10px"></div>
pub const AQUAMARINE2: Rgb = Rgb { r: 118, g: 238, b: 198 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#66CDAA;margin: 10px 10px 10px 10px"></div>
pub const AQUAMARINE3: Rgb = Rgb { r: 102, g: 205, b: 170 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#458B74;margin: 10px 10px 10px 10px"></div>
pub const AQUAMARINE4: Rgb = Rgb { r: 69, g: 139, b: 116 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#C1FFC1;margin: 10px 10px 10px 10px"></div>
pub const DARKSEAGREEN1: Rgb = Rgb { r: 193, g: 255, b: 193 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B4EEB4;margin: 10px 10px 10px 10px"></div>
pub const DARKSEAGREEN2: Rgb = Rgb { r: 180, g: 238, b: 180 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#9BCD9B;margin: 10px 10px 10px 10px"></div>
pub const DARKSEAGREEN3: Rgb = Rgb { r: 155, g: 205, b: 155 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#698B69;margin: 10px 10px 10px 10px"></div>
pub const DARKSEAGREEN4: Rgb = Rgb { r: 105, g: 139, b: 105 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#54FF9F;margin: 10px 10px 10px 10px"></div>
pub const SEAGREEN1: Rgb = Rgb { r: 84, g: 255, b: 159 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#4EEE94;margin: 10px 10px 10px 10px"></div>
pub const SEAGREEN2: Rgb = Rgb { r: 78, g: 238, b: 148 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#43CD80;margin: 10px 10px 10px 10px"></div>
pub const SEAGREEN3: Rgb = Rgb { r: 67, g: 205, b: 128 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#2E8B57;margin: 10px 10px 10px 10px"></div>
pub const SEAGREEN4: Rgb = Rgb { r: 46, g: 139, b: 87 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#9AFF9A;margin: 10px 10px 10px 10px"></div>
pub const PALEGREEN1: Rgb = Rgb { r: 154, g: 255, b: 154 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#90EE90;margin: 10px 10px 10px 10px"></div>
pub const PALEGREEN2: Rgb = Rgb { r: 144, g: 238, b: 144 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7CCD7C;margin: 10px 10px 10px 10px"></div>
pub const PALEGREEN3: Rgb = Rgb { r: 124, g: 205, b: 124 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#548B54;margin: 10px 10px 10px 10px"></div>
pub const PALEGREEN4: Rgb = Rgb { r: 84, g: 139, b: 84 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00FF7F;margin: 10px 10px 10px 10px"></div>
pub const SPRINGGREEN1: Rgb = Rgb { r: 0, g: 255, b: 127 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00EE76;margin: 10px 10px 10px 10px"></div>
pub const SPRINGGREEN2: Rgb = Rgb { r: 0, g: 238, b: 118 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00CD66;margin: 10px 10px 10px 10px"></div>
pub const SPRINGGREEN3: Rgb = Rgb { r: 0, g: 205, b: 102 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#008B45;margin: 10px 10px 10px 10px"></div>
pub const SPRINGGREEN4: Rgb = Rgb { r: 0, g: 139, b: 69 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00FF00;margin: 10px 10px 10px 10px"></div>
pub const GREEN1: Rgb = Rgb { r: 0, g: 255, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00EE00;margin: 10px 10px 10px 10px"></div>
pub const GREEN2: Rgb = Rgb { r: 0, g: 238, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00CD00;margin: 10px 10px 10px 10px"></div>
pub const GREEN3: Rgb = Rgb { r: 0, g: 205, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#008B00;margin: 10px 10px 10px 10px"></div>
pub const GREEN4: Rgb = Rgb { r: 0, g: 139, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7FFF00;margin: 10px 10px 10px 10px"></div>
pub const CHARTREUSE1: Rgb = Rgb { r: 127, g: 255, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#76EE00;margin: 10px 10px 10px 10px"></div>
pub const CHARTREUSE2: Rgb = Rgb { r: 118, g: 238, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#66CD00;margin: 10px 10px 10px 10px"></div>
pub const CHARTREUSE3: Rgb = Rgb { r: 102, g: 205, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#458B00;margin: 10px 10px 10px 10px"></div>
pub const CHARTREUSE4: Rgb = Rgb { r: 69, g: 139, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#C0FF3E;margin: 10px 10px 10px 10px"></div>
pub const OLIVEDRAB1: Rgb = Rgb { r: 192, g: 255, b: 62 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B3EE3A;margin: 10px 10px 10px 10px"></div>
pub const OLIVEDRAB2: Rgb = Rgb { r: 179, g: 238, b: 58 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#9ACD32;margin: 10px 10px 10px 10px"></div>
pub const OLIVEDRAB3: Rgb = Rgb { r: 154, g: 205, b: 50 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#698B22;margin: 10px 10px 10px 10px"></div>
pub const OLIVEDRAB4: Rgb = Rgb { r: 105, g: 139, b: 34 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CAFF70;margin: 10px 10px 10px 10px"></div>
pub const DARKOLIVEGREEN1: Rgb = Rgb { r: 202, g: 255, b: 112 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#BCEE68;margin: 10px 10px 10px 10px"></div>
pub const DARKOLIVEGREEN2: Rgb = Rgb { r: 188, g: 238, b: 104 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#A2CD5A;margin: 10px 10px 10px 10px"></div>
pub const DARKOLIVEGREEN3: Rgb = Rgb { r: 162, g: 205, b: 90 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#6E8B3D;margin: 10px 10px 10px 10px"></div>
pub const DARKOLIVEGREEN4: Rgb = Rgb { r: 110, g: 139, b: 61 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFF68F;margin: 10px 10px 10px 10px"></div>
pub const KHAKI1: Rgb = Rgb { r: 255, g: 247, b: 143 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEE685;margin: 10px 10px 10px 10px"></div>
pub const KHAKI2: Rgb = Rgb { r: 238, g: 230, b: 133 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDC673;margin: 10px 10px 10px 10px"></div>
pub const KHAKI3: Rgb = Rgb { r: 205, g: 198, b: 115 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B864E;margin: 10px 10px 10px 10px"></div>
pub const KHAKI4: Rgb = Rgb { r: 139, g: 134, b: 78 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFEC8B;margin: 10px 10px 10px 10px"></div>
pub const LIGHTGOLDENROD1: Rgb = Rgb { r: 255, g: 236, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEDC82;margin: 10px 10px 10px 10px"></div>
pub const LIGHTGOLDENROD2: Rgb = Rgb { r: 238, g: 220, b: 130 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDBE70;margin: 10px 10px 10px 10px"></div>
pub const LIGHTGOLDENROD3: Rgb = Rgb { r: 205, g: 190, b: 112 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B814C;margin: 10px 10px 10px 10px"></div>
pub const LIGHTGOLDENROD4: Rgb = Rgb { r: 139, g: 129, b: 76 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFFFE0;margin: 10px 10px 10px 10px"></div>
pub const LIGHTYELLOW1: Rgb = Rgb { r: 255, g: 255, b: 224 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEEED1;margin: 10px 10px 10px 10px"></div>
pub const LIGHTYELLOW2: Rgb = Rgb { r: 238, g: 238, b: 209 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDCDB4;margin: 10px 10px 10px 10px"></div>
pub const LIGHTYELLOW3: Rgb = Rgb { r: 205, g: 205, b: 180 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B8B7A;margin: 10px 10px 10px 10px"></div>
pub const LIGHTYELLOW4: Rgb = Rgb { r: 139, g: 139, b: 122 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFFF00;margin: 10px 10px 10px 10px"></div>
pub const YELLOW1: Rgb = Rgb { r: 255, g: 255, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEEE00;margin: 10px 10px 10px 10px"></div>
pub const YELLOW2: Rgb = Rgb { r: 238, g: 238, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDCD00;margin: 10px 10px 10px 10px"></div>
pub const YELLOW3: Rgb = Rgb { r: 205, g: 205, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B8B00;margin: 10px 10px 10px 10px"></div>
pub const YELLOW4: Rgb = Rgb { r: 139, g: 139, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFD700;margin: 10px 10px 10px 10px"></div>
pub const GOLD1: Rgb = Rgb { r: 255, g: 215, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEC900;margin: 10px 10px 10px 10px"></div>
pub const GOLD2: Rgb = Rgb { r: 238, g: 201, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDAD00;margin: 10px 10px 10px 10px"></div>
pub const GOLD3: Rgb = Rgb { r: 205, g: 173, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B7500;margin: 10px 10px 10px 10px"></div>
pub const GOLD4: Rgb = Rgb { r: 139, g: 117, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFC125;margin: 10px 10px 10px 10px"></div>
pub const GOLDENROD1: Rgb = Rgb { r: 255, g: 193, b: 37 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEB422;margin: 10px 10px 10px 10px"></div>
pub const GOLDENROD2: Rgb = Rgb { r: 238, g: 180, b: 34 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD9B1D;margin: 10px 10px 10px 10px"></div>
pub const GOLDENROD3: Rgb = Rgb { r: 205, g: 155, b: 29 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B6914;margin: 10px 10px 10px 10px"></div>
pub const GOLDENROD4: Rgb = Rgb { r: 139, g: 105, b: 19 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFB90F;margin: 10px 10px 10px 10px"></div>
pub const DARKGOLDENROD1: Rgb = Rgb { r: 255, g: 185, b: 15 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEAD0E;margin: 10px 10px 10px 10px"></div>
pub const DARKGOLDENROD2: Rgb = Rgb { r: 238, g: 173, b: 14 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD950C;margin: 10px 10px 10px 10px"></div>
pub const DARKGOLDENROD3: Rgb = Rgb { r: 205, g: 149, b: 12 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B6508;margin: 10px 10px 10px 10px"></div>
pub const DARKGOLDENROD4: Rgb = Rgb { r: 139, g: 101, b: 7 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFC1C1;margin: 10px 10px 10px 10px"></div>
pub const ROSYBROWN1: Rgb = Rgb { r: 255, g: 193, b: 193 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEB4B4;margin: 10px 10px 10px 10px"></div>
pub const ROSYBROWN2: Rgb = Rgb { r: 238, g: 180, b: 180 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD9B9B;margin: 10px 10px 10px 10px"></div>
pub const ROSYBROWN3: Rgb = Rgb { r: 205, g: 155, b: 155 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B6969;margin: 10px 10px 10px 10px"></div>
pub const ROSYBROWN4: Rgb = Rgb { r: 139, g: 105, b: 105 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF6A6A;margin: 10px 10px 10px 10px"></div>
pub const INDIANRED1: Rgb = Rgb { r: 255, g: 106, b: 106 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE6363;margin: 10px 10px 10px 10px"></div>
pub const INDIANRED2: Rgb = Rgb { r: 238, g: 99, b: 99 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD5555;margin: 10px 10px 10px 10px"></div>
pub const INDIANRED3: Rgb = Rgb { r: 205, g: 85, b: 85 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B3A3A;margin: 10px 10px 10px 10px"></div>
pub const INDIANRED4: Rgb = Rgb { r: 139, g: 58, b: 58 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF8247;margin: 10px 10px 10px 10px"></div>
pub const SIENNA1: Rgb = Rgb { r: 255, g: 130, b: 71 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE7942;margin: 10px 10px 10px 10px"></div>
pub const SIENNA2: Rgb = Rgb { r: 238, g: 121, b: 66 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD6839;margin: 10px 10px 10px 10px"></div>
pub const SIENNA3: Rgb = Rgb { r: 205, g: 104, b: 57 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B4726;margin: 10px 10px 10px 10px"></div>
pub const SIENNA4: Rgb = Rgb { r: 139, g: 71, b: 38 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFD39B;margin: 10px 10px 10px 10px"></div>
pub const BURLYWOOD1: Rgb = Rgb { r: 255, g: 211, b: 155 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEC591;margin: 10px 10px 10px 10px"></div>
pub const BURLYWOOD2: Rgb = Rgb { r: 238, g: 197, b: 145 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDAA7D;margin: 10px 10px 10px 10px"></div>
pub const BURLYWOOD3: Rgb = Rgb { r: 205, g: 170, b: 125 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B7355;margin: 10px 10px 10px 10px"></div>
pub const BURLYWOOD4: Rgb = Rgb { r: 139, g: 115, b: 85 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFE7BA;margin: 10px 10px 10px 10px"></div>
pub const WHEAT1: Rgb = Rgb { r: 255, g: 231, b: 186 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EED8AE;margin: 10px 10px 10px 10px"></div>
pub const WHEAT2: Rgb = Rgb { r: 238, g: 216, b: 174 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDBA96;margin: 10px 10px 10px 10px"></div>
pub const WHEAT3: Rgb = Rgb { r: 205, g: 186, b: 150 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B7E66;margin: 10px 10px 10px 10px"></div>
pub const WHEAT4: Rgb = Rgb { r: 139, g: 126, b: 102 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFA54F;margin: 10px 10px 10px 10px"></div>
pub const TAN1: Rgb = Rgb { r: 255, g: 165, b: 79 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE9A49;margin: 10px 10px 10px 10px"></div>
pub const TAN2: Rgb = Rgb { r: 238, g: 154, b: 73 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD853F;margin: 10px 10px 10px 10px"></div>
pub const TAN3: Rgb = Rgb { r: 205, g: 133, b: 63 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B5A2B;margin: 10px 10px 10px 10px"></div>
pub const TAN4: Rgb = Rgb { r: 139, g: 90, b: 43 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF7F24;margin: 10px 10px 10px 10px"></div>
pub const CHOCOLATE1: Rgb = Rgb { r: 255, g: 127, b: 36 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE7621;margin: 10px 10px 10px 10px"></div>
pub const CHOCOLATE2: Rgb = Rgb { r: 238, g: 118, b: 33 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD661D;margin: 10px 10px 10px 10px"></div>
pub const CHOCOLATE3: Rgb = Rgb { r: 205, g: 102, b: 29 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B4513;margin: 10px 10px 10px 10px"></div>
pub const CHOCOLATE4: Rgb = Rgb { r: 139, g: 69, b: 19 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF3030;margin: 10px 10px 10px 10px"></div>
pub const FIREBRICK1: Rgb = Rgb { r: 255, g: 48, b: 48 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE2C2C;margin: 10px 10px 10px 10px"></div>
pub const FIREBRICK2: Rgb = Rgb { r: 238, g: 44, b: 44 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD2626;margin: 10px 10px 10px 10px"></div>
pub const FIREBRICK3: Rgb = Rgb { r: 205, g: 38, b: 38 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B1A1A;margin: 10px 10px 10px 10px"></div>
pub const FIREBRICK4: Rgb = Rgb { r: 139, g: 26, b: 26 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF4040;margin: 10px 10px 10px 10px"></div>
pub const BROWN1: Rgb = Rgb { r: 255, g: 64, b: 64 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE3B3B;margin: 10px 10px 10px 10px"></div>
pub const BROWN2: Rgb = Rgb { r: 238, g: 59, b: 59 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD3333;margin: 10px 10px 10px 10px"></div>
pub const BROWN3: Rgb = Rgb { r: 205, g: 51, b: 51 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B2323;margin: 10px 10px 10px 10px"></div>
pub const BROWN4: Rgb = Rgb { r: 139, g: 35, b: 35 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF8C69;margin: 10px 10px 10px 10px"></div>
pub const SALMON1: Rgb = Rgb { r: 255, g: 140, b: 105 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE8262;margin: 10px 10px 10px 10px"></div>
pub const SALMON2: Rgb = Rgb { r: 238, g: 130, b: 98 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD7054;margin: 10px 10px 10px 10px"></div>
pub const SALMON3: Rgb = Rgb { r: 205, g: 112, b: 84 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B4C39;margin: 10px 10px 10px 10px"></div>
pub const SALMON4: Rgb = Rgb { r: 139, g: 76, b: 57 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFA07A;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSALMON1: Rgb = Rgb { r: 255, g: 160, b: 122 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE9572;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSALMON2: Rgb = Rgb { r: 238, g: 149, b: 114 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD8162;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSALMON3: Rgb = Rgb { r: 205, g: 129, b: 98 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B5742;margin: 10px 10px 10px 10px"></div>
pub const LIGHTSALMON4: Rgb = Rgb { r: 139, g: 87, b: 66 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFA500;margin: 10px 10px 10px 10px"></div>
pub const ORANGE1: Rgb = Rgb { r: 255, g: 165, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE9A00;margin: 10px 10px 10px 10px"></div>
pub const ORANGE2: Rgb = Rgb { r: 238, g: 154, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD8500;margin: 10px 10px 10px 10px"></div>
pub const ORANGE3: Rgb = Rgb { r: 205, g: 133, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B5A00;margin: 10px 10px 10px 10px"></div>
pub const ORANGE4: Rgb = Rgb { r: 139, g: 90, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF7F00;margin: 10px 10px 10px 10px"></div>
pub const DARKORANGE1: Rgb = Rgb { r: 255, g: 127, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE7600;margin: 10px 10px 10px 10px"></div>
pub const DARKORANGE2: Rgb = Rgb { r: 238, g: 118, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD6600;margin: 10px 10px 10px 10px"></div>
pub const DARKORANGE3: Rgb = Rgb { r: 205, g: 102, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B4500;margin: 10px 10px 10px 10px"></div>
pub const DARKORANGE4: Rgb = Rgb { r: 139, g: 69, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF7256;margin: 10px 10px 10px 10px"></div>
pub const CORAL1: Rgb = Rgb { r: 255, g: 114, b: 86 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE6A50;margin: 10px 10px 10px 10px"></div>
pub const CORAL2: Rgb = Rgb { r: 238, g: 106, b: 80 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD5B45;margin: 10px 10px 10px 10px"></div>
pub const CORAL3: Rgb = Rgb { r: 205, g: 91, b: 69 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B3E2F;margin: 10px 10px 10px 10px"></div>
pub const CORAL4: Rgb = Rgb { r: 139, g: 62, b: 47 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF6347;margin: 10px 10px 10px 10px"></div>
pub const TOMATO1: Rgb = Rgb { r: 255, g: 99, b: 71 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE5C42;margin: 10px 10px 10px 10px"></div>
pub const TOMATO2: Rgb = Rgb { r: 238, g: 92, b: 66 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD4F39;margin: 10px 10px 10px 10px"></div>
pub const TOMATO3: Rgb = Rgb { r: 205, g: 79, b: 57 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B3626;margin: 10px 10px 10px 10px"></div>
pub const TOMATO4: Rgb = Rgb { r: 139, g: 54, b: 38 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF4500;margin: 10px 10px 10px 10px"></div>
pub const ORANGERED1: Rgb = Rgb { r: 255, g: 69, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE4000;margin: 10px 10px 10px 10px"></div>
pub const ORANGERED2: Rgb = Rgb { r: 238, g: 64, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD3700;margin: 10px 10px 10px 10px"></div>
pub const ORANGERED3: Rgb = Rgb { r: 205, g: 55, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B2500;margin: 10px 10px 10px 10px"></div>
pub const ORANGERED4: Rgb = Rgb { r: 139, g: 37, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF0000;margin: 10px 10px 10px 10px"></div>
pub const RED1: Rgb = Rgb { r: 255, g: 0, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE0000;margin: 10px 10px 10px 10px"></div>
pub const RED2: Rgb = Rgb { r: 238, g: 0, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD0000;margin: 10px 10px 10px 10px"></div>
pub const RED3: Rgb = Rgb { r: 205, g: 0, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B0000;margin: 10px 10px 10px 10px"></div>
pub const RED4: Rgb = Rgb { r: 139, g: 0, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF1493;margin: 10px 10px 10px 10px"></div>
pub const DEEPPINK1: Rgb = Rgb { r: 255, g: 19, b: 147 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE1289;margin: 10px 10px 10px 10px"></div>
pub const DEEPPINK2: Rgb = Rgb { r: 238, g: 18, b: 137 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD1076;margin: 10px 10px 10px 10px"></div>
pub const DEEPPINK3: Rgb = Rgb { r: 205, g: 16, b: 118 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B0A50;margin: 10px 10px 10px 10px"></div>
pub const DEEPPINK4: Rgb = Rgb { r: 139, g: 9, b: 80 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF6EB4;margin: 10px 10px 10px 10px"></div>
pub const HOTPINK1: Rgb = Rgb { r: 255, g: 110, b: 180 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE6AA7;margin: 10px 10px 10px 10px"></div>
pub const HOTPINK2: Rgb = Rgb { r: 238, g: 106, b: 167 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD6090;margin: 10px 10px 10px 10px"></div>
pub const HOTPINK3: Rgb = Rgb { r: 205, g: 96, b: 144 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B3A62;margin: 10px 10px 10px 10px"></div>
pub const HOTPINK4: Rgb = Rgb { r: 139, g: 58, b: 98 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFB5C5;margin: 10px 10px 10px 10px"></div>
pub const PINK1: Rgb = Rgb { r: 255, g: 181, b: 197 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEA9B8;margin: 10px 10px 10px 10px"></div>
pub const PINK2: Rgb = Rgb { r: 238, g: 169, b: 184 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD919E;margin: 10px 10px 10px 10px"></div>
pub const PINK3: Rgb = Rgb { r: 205, g: 145, b: 158 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B636C;margin: 10px 10px 10px 10px"></div>
pub const PINK4: Rgb = Rgb { r: 139, g: 99, b: 108 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFAEB9;margin: 10px 10px 10px 10px"></div>
pub const LIGHTPINK1: Rgb = Rgb { r: 255, g: 174, b: 185 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEA2AD;margin: 10px 10px 10px 10px"></div>
pub const LIGHTPINK2: Rgb = Rgb { r: 238, g: 162, b: 173 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD8C95;margin: 10px 10px 10px 10px"></div>
pub const LIGHTPINK3: Rgb = Rgb { r: 205, g: 140, b: 149 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B5F65;margin: 10px 10px 10px 10px"></div>
pub const LIGHTPINK4: Rgb = Rgb { r: 139, g: 95, b: 101 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF82AB;margin: 10px 10px 10px 10px"></div>
pub const PALEVIOLETRED1: Rgb = Rgb { r: 255, g: 130, b: 171 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE799F;margin: 10px 10px 10px 10px"></div>
pub const PALEVIOLETRED2: Rgb = Rgb { r: 238, g: 121, b: 159 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD6889;margin: 10px 10px 10px 10px"></div>
pub const PALEVIOLETRED3: Rgb = Rgb { r: 205, g: 104, b: 137 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B475D;margin: 10px 10px 10px 10px"></div>
pub const PALEVIOLETRED4: Rgb = Rgb { r: 139, g: 71, b: 93 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF34B3;margin: 10px 10px 10px 10px"></div>
pub const MAROON1: Rgb = Rgb { r: 255, g: 52, b: 179 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE30A7;margin: 10px 10px 10px 10px"></div>
pub const MAROON2: Rgb = Rgb { r: 238, g: 48, b: 167 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD2990;margin: 10px 10px 10px 10px"></div>
pub const MAROON3: Rgb = Rgb { r: 205, g: 41, b: 144 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B1C62;margin: 10px 10px 10px 10px"></div>
pub const MAROON4: Rgb = Rgb { r: 139, g: 28, b: 98 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF3E96;margin: 10px 10px 10px 10px"></div>
pub const VIOLETRED1: Rgb = Rgb { r: 255, g: 62, b: 150 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE3A8C;margin: 10px 10px 10px 10px"></div>
pub const VIOLETRED2: Rgb = Rgb { r: 238, g: 58, b: 140 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD3278;margin: 10px 10px 10px 10px"></div>
pub const VIOLETRED3: Rgb = Rgb { r: 205, g: 50, b: 120 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B2252;margin: 10px 10px 10px 10px"></div>
pub const VIOLETRED4: Rgb = Rgb { r: 139, g: 34, b: 82 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF00FF;margin: 10px 10px 10px 10px"></div>
pub const MAGENTA1: Rgb = Rgb { r: 255, g: 0, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE00EE;margin: 10px 10px 10px 10px"></div>
pub const MAGENTA2: Rgb = Rgb { r: 238, g: 0, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD00CD;margin: 10px 10px 10px 10px"></div>
pub const MAGENTA3: Rgb = Rgb { r: 205, g: 0, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B008B;margin: 10px 10px 10px 10px"></div>
pub const MAGENTA4: Rgb = Rgb { r: 139, g: 0, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FF83FA;margin: 10px 10px 10px 10px"></div>
pub const ORCHID1: Rgb = Rgb { r: 255, g: 131, b: 250 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EE7AE9;margin: 10px 10px 10px 10px"></div>
pub const ORCHID2: Rgb = Rgb { r: 238, g: 122, b: 233 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD69C9;margin: 10px 10px 10px 10px"></div>
pub const ORCHID3: Rgb = Rgb { r: 205, g: 105, b: 201 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B4789;margin: 10px 10px 10px 10px"></div>
pub const ORCHID4: Rgb = Rgb { r: 139, g: 71, b: 137 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFBBFF;margin: 10px 10px 10px 10px"></div>
pub const PLUM1: Rgb = Rgb { r: 255, g: 187, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EEAEEE;margin: 10px 10px 10px 10px"></div>
pub const PLUM2: Rgb = Rgb { r: 238, g: 174, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CD96CD;margin: 10px 10px 10px 10px"></div>
pub const PLUM3: Rgb = Rgb { r: 205, g: 150, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B668B;margin: 10px 10px 10px 10px"></div>
pub const PLUM4: Rgb = Rgb { r: 139, g: 102, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#E066FF;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMORCHID1: Rgb = Rgb { r: 224, g: 102, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#D15FEE;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMORCHID2: Rgb = Rgb { r: 209, g: 95, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B452CD;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMORCHID3: Rgb = Rgb { r: 180, g: 82, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7A378B;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMORCHID4: Rgb = Rgb { r: 122, g: 55, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#BF3EFF;margin: 10px 10px 10px 10px"></div>
pub const DARKORCHID1: Rgb = Rgb { r: 191, g: 62, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B23AEE;margin: 10px 10px 10px 10px"></div>
pub const DARKORCHID2: Rgb = Rgb { r: 178, g: 58, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#9A32CD;margin: 10px 10px 10px 10px"></div>
pub const DARKORCHID3: Rgb = Rgb { r: 154, g: 50, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#68228B;margin: 10px 10px 10px 10px"></div>
pub const DARKORCHID4: Rgb = Rgb { r: 104, g: 34, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#9B30FF;margin: 10px 10px 10px 10px"></div>
pub const PURPLE1: Rgb = Rgb { r: 155, g: 48, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#912CEE;margin: 10px 10px 10px 10px"></div>
pub const PURPLE2: Rgb = Rgb { r: 145, g: 44, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7D26CD;margin: 10px 10px 10px 10px"></div>
pub const PURPLE3: Rgb = Rgb { r: 125, g: 38, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#551A8B;margin: 10px 10px 10px 10px"></div>
pub const PURPLE4: Rgb = Rgb { r: 85, g: 26, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#AB82FF;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMPURPLE1: Rgb = Rgb { r: 171, g: 130, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#9F79EE;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMPURPLE2: Rgb = Rgb { r: 159, g: 121, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8968CD;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMPURPLE3: Rgb = Rgb { r: 137, g: 104, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#5D478B;margin: 10px 10px 10px 10px"></div>
pub const MEDIUMPURPLE4: Rgb = Rgb { r: 93, g: 71, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFE1FF;margin: 10px 10px 10px 10px"></div>
pub const THISTLE1: Rgb = Rgb { r: 255, g: 225, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EED2EE;margin: 10px 10px 10px 10px"></div>
pub const THISTLE2: Rgb = Rgb { r: 238, g: 210, b: 238 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CDB5CD;margin: 10px 10px 10px 10px"></div>
pub const THISTLE3: Rgb = Rgb { r: 205, g: 181, b: 205 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B7B8B;margin: 10px 10px 10px 10px"></div>
pub const THISTLE4: Rgb = Rgb { r: 139, g: 123, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#000000;margin: 10px 10px 10px 10px"></div>
pub const GREY0: Rgb = Rgb { r: 0, g: 0, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#030303;margin: 10px 10px 10px 10px"></div>
pub const GREY1: Rgb = Rgb { r: 3, g: 3, b: 3 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#050505;margin: 10px 10px 10px 10px"></div>
pub const GREY2: Rgb = Rgb { r: 5, g: 5, b: 5 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#080808;margin: 10px 10px 10px 10px"></div>
pub const GREY3: Rgb = Rgb { r: 7, g: 7, b: 7 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#0A0A0A;margin: 10px 10px 10px 10px"></div>
pub const GREY4: Rgb = Rgb { r: 9, g: 9, b: 9 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#0D0D0D;margin: 10px 10px 10px 10px"></div>
pub const GREY5: Rgb = Rgb { r: 13, g: 13, b: 13 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#0F0F0F;margin: 10px 10px 10px 10px"></div>
pub const GREY6: Rgb = Rgb { r: 15, g: 15, b: 15 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#121212;margin: 10px 10px 10px 10px"></div>
pub const GREY7: Rgb = Rgb { r: 18, g: 18, b: 18 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#141414;margin: 10px 10px 10px 10px"></div>
pub const GREY8: Rgb = Rgb { r: 19, g: 19, b: 19 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#171717;margin: 10px 10px 10px 10px"></div>
pub const GREY9: Rgb = Rgb { r: 23, g: 23, b: 23 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#1A1A1A;margin: 10px 10px 10px 10px"></div>
pub const GREY10: Rgb = Rgb { r: 26, g: 26, b: 26 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#1C1C1C;margin: 10px 10px 10px 10px"></div>
pub const GREY11: Rgb = Rgb { r: 28, g: 28, b: 28 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#1F1F1F;margin: 10px 10px 10px 10px"></div>
pub const GREY12: Rgb = Rgb { r: 31, g: 31, b: 31 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#212121;margin: 10px 10px 10px 10px"></div>
pub const GREY13: Rgb = Rgb { r: 33, g: 33, b: 33 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#242424;margin: 10px 10px 10px 10px"></div>
pub const GREY14: Rgb = Rgb { r: 36, g: 36, b: 36 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#262626;margin: 10px 10px 10px 10px"></div>
pub const GREY15: Rgb = Rgb { r: 38, g: 38, b: 38 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#292929;margin: 10px 10px 10px 10px"></div>
pub const GREY16: Rgb = Rgb { r: 41, g: 41, b: 41 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#2B2B2B;margin: 10px 10px 10px 10px"></div>
pub const GREY17: Rgb = Rgb { r: 43, g: 43, b: 43 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#2E2E2E;margin: 10px 10px 10px 10px"></div>
pub const GREY18: Rgb = Rgb { r: 46, g: 46, b: 46 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#303030;margin: 10px 10px 10px 10px"></div>
pub const GREY19: Rgb = Rgb { r: 48, g: 48, b: 48 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#333333;margin: 10px 10px 10px 10px"></div>
pub const GREY20: Rgb = Rgb { r: 51, g: 51, b: 51 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#363636;margin: 10px 10px 10px 10px"></div>
pub const GREY21: Rgb = Rgb { r: 54, g: 54, b: 54 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#383838;margin: 10px 10px 10px 10px"></div>
pub const GREY22: Rgb = Rgb { r: 56, g: 56, b: 56 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#3B3B3B;margin: 10px 10px 10px 10px"></div>
pub const GREY23: Rgb = Rgb { r: 59, g: 59, b: 59 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#3D3D3D;margin: 10px 10px 10px 10px"></div>
pub const GREY24: Rgb = Rgb { r: 61, g: 61, b: 61 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#404040;margin: 10px 10px 10px 10px"></div>
pub const GREY25: Rgb = Rgb { r: 64, g: 64, b: 64 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#424242;margin: 10px 10px 10px 10px"></div>
pub const GREY26: Rgb = Rgb { r: 66, g: 66, b: 66 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#454545;margin: 10px 10px 10px 10px"></div>
pub const GREY27: Rgb = Rgb { r: 69, g: 69, b: 69 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#474747;margin: 10px 10px 10px 10px"></div>
pub const GREY28: Rgb = Rgb { r: 71, g: 71, b: 71 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#4A4A4A;margin: 10px 10px 10px 10px"></div>
pub const GREY29: Rgb = Rgb { r: 74, g: 74, b: 74 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#4D4D4D;margin: 10px 10px 10px 10px"></div>
pub const GREY30: Rgb = Rgb { r: 77, g: 77, b: 77 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#4F4F4F;margin: 10px 10px 10px 10px"></div>
pub const GREY31: Rgb = Rgb { r: 79, g: 79, b: 79 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#525252;margin: 10px 10px 10px 10px"></div>
pub const GREY32: Rgb = Rgb { r: 82, g: 82, b: 82 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#545454;margin: 10px 10px 10px 10px"></div>
pub const GREY33: Rgb = Rgb { r: 84, g: 84, b: 84 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#575757;margin: 10px 10px 10px 10px"></div>
pub const GREY34: Rgb = Rgb { r: 87, g: 87, b: 87 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#595959;margin: 10px 10px 10px 10px"></div>
pub const GREY35: Rgb = Rgb { r: 89, g: 89, b: 89 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#5C5C5C;margin: 10px 10px 10px 10px"></div>
pub const GREY36: Rgb = Rgb { r: 92, g: 92, b: 92 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#5E5E5E;margin: 10px 10px 10px 10px"></div>
pub const GREY37: Rgb = Rgb { r: 94, g: 94, b: 94 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#616161;margin: 10px 10px 10px 10px"></div>
pub const GREY38: Rgb = Rgb { r: 97, g: 97, b: 97 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#636363;margin: 10px 10px 10px 10px"></div>
pub const GREY39: Rgb = Rgb { r: 99, g: 99, b: 99 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#666666;margin: 10px 10px 10px 10px"></div>
pub const GREY40: Rgb = Rgb { r: 102, g: 102, b: 102 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#696969;margin: 10px 10px 10px 10px"></div>
pub const GREY41: Rgb = Rgb { r: 105, g: 105, b: 105 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#6B6B6B;margin: 10px 10px 10px 10px"></div>
pub const GREY42: Rgb = Rgb { r: 107, g: 107, b: 107 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#6E6E6E;margin: 10px 10px 10px 10px"></div>
pub const GREY43: Rgb = Rgb { r: 110, g: 110, b: 110 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#707070;margin: 10px 10px 10px 10px"></div>
pub const GREY44: Rgb = Rgb { r: 112, g: 112, b: 112 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#737373;margin: 10px 10px 10px 10px"></div>
pub const GREY45: Rgb = Rgb { r: 115, g: 115, b: 115 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#757575;margin: 10px 10px 10px 10px"></div>
pub const GREY46: Rgb = Rgb { r: 117, g: 117, b: 117 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#787878;margin: 10px 10px 10px 10px"></div>
pub const GREY47: Rgb = Rgb { r: 120, g: 120, b: 120 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7A7A7A;margin: 10px 10px 10px 10px"></div>
pub const GREY48: Rgb = Rgb { r: 122, g: 122, b: 122 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7D7D7D;margin: 10px 10px 10px 10px"></div>
pub const GREY49: Rgb = Rgb { r: 125, g: 125, b: 125 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#7F7F7F;margin: 10px 10px 10px 10px"></div>
pub const GREY50: Rgb = Rgb { r: 127, g: 127, b: 127 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#828282;margin: 10px 10px 10px 10px"></div>
pub const GREY51: Rgb = Rgb { r: 130, g: 130, b: 130 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#858585;margin: 10px 10px 10px 10px"></div>
pub const GREY52: Rgb = Rgb { r: 133, g: 133, b: 133 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#878787;margin: 10px 10px 10px 10px"></div>
pub const GREY53: Rgb = Rgb { r: 135, g: 135, b: 135 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8A8A8A;margin: 10px 10px 10px 10px"></div>
pub const GREY54: Rgb = Rgb { r: 138, g: 138, b: 138 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8C8C8C;margin: 10px 10px 10px 10px"></div>
pub const GREY55: Rgb = Rgb { r: 140, g: 140, b: 140 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8F8F8F;margin: 10px 10px 10px 10px"></div>
pub const GREY56: Rgb = Rgb { r: 143, g: 143, b: 143 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#919191;margin: 10px 10px 10px 10px"></div>
pub const GREY57: Rgb = Rgb { r: 145, g: 145, b: 145 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#949494;margin: 10px 10px 10px 10px"></div>
pub const GREY58: Rgb = Rgb { r: 148, g: 148, b: 148 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#969696;margin: 10px 10px 10px 10px"></div>
pub const GREY59: Rgb = Rgb { r: 150, g: 150, b: 150 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#999999;margin: 10px 10px 10px 10px"></div>
pub const GREY60: Rgb = Rgb { r: 153, g: 153, b: 153 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#9C9C9C;margin: 10px 10px 10px 10px"></div>
pub const GREY61: Rgb = Rgb { r: 156, g: 156, b: 156 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#9E9E9E;margin: 10px 10px 10px 10px"></div>
pub const GREY62: Rgb = Rgb { r: 158, g: 158, b: 158 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#A1A1A1;margin: 10px 10px 10px 10px"></div>
pub const GREY63: Rgb = Rgb { r: 161, g: 161, b: 161 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#A3A3A3;margin: 10px 10px 10px 10px"></div>
pub const GREY64: Rgb = Rgb { r: 163, g: 163, b: 163 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#A6A6A6;margin: 10px 10px 10px 10px"></div>
pub const GREY65: Rgb = Rgb { r: 166, g: 166, b: 166 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#A8A8A8;margin: 10px 10px 10px 10px"></div>
pub const GREY66: Rgb = Rgb { r: 168, g: 168, b: 168 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#ABABAB;margin: 10px 10px 10px 10px"></div>
pub const GREY67: Rgb = Rgb { r: 171, g: 171, b: 171 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#ADADAD;margin: 10px 10px 10px 10px"></div>
pub const GREY68: Rgb = Rgb { r: 173, g: 173, b: 173 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B0B0B0;margin: 10px 10px 10px 10px"></div>
pub const GREY69: Rgb = Rgb { r: 176, g: 176, b: 176 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B3B3B3;margin: 10px 10px 10px 10px"></div>
pub const GREY70: Rgb = Rgb { r: 179, g: 179, b: 179 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B5B5B5;margin: 10px 10px 10px 10px"></div>
pub const GREY71: Rgb = Rgb { r: 181, g: 181, b: 181 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#B8B8B8;margin: 10px 10px 10px 10px"></div>
pub const GREY72: Rgb = Rgb { r: 184, g: 184, b: 184 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#BABABA;margin: 10px 10px 10px 10px"></div>
pub const GREY73: Rgb = Rgb { r: 186, g: 186, b: 186 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#BDBDBD;margin: 10px 10px 10px 10px"></div>
pub const GREY74: Rgb = Rgb { r: 189, g: 189, b: 189 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#BFBFBF;margin: 10px 10px 10px 10px"></div>
pub const GREY75: Rgb = Rgb { r: 191, g: 191, b: 191 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#C2C2C2;margin: 10px 10px 10px 10px"></div>
pub const GREY76: Rgb = Rgb { r: 194, g: 194, b: 194 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#C4C4C4;margin: 10px 10px 10px 10px"></div>
pub const GREY77: Rgb = Rgb { r: 196, g: 196, b: 196 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#C7C7C7;margin: 10px 10px 10px 10px"></div>
pub const GREY78: Rgb = Rgb { r: 199, g: 199, b: 199 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#C9C9C9;margin: 10px 10px 10px 10px"></div>
pub const GREY79: Rgb = Rgb { r: 201, g: 201, b: 201 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CCCCCC;margin: 10px 10px 10px 10px"></div>
pub const GREY80: Rgb = Rgb { r: 204, g: 204, b: 204 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#CFCFCF;margin: 10px 10px 10px 10px"></div>
pub const GREY81: Rgb = Rgb { r: 207, g: 207, b: 207 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#D1D1D1;margin: 10px 10px 10px 10px"></div>
pub const GREY82: Rgb = Rgb { r: 209, g: 209, b: 209 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#D4D4D4;margin: 10px 10px 10px 10px"></div>
pub const GREY83: Rgb = Rgb { r: 212, g: 212, b: 212 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#D6D6D6;margin: 10px 10px 10px 10px"></div>
pub const GREY84: Rgb = Rgb { r: 214, g: 214, b: 214 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#D9D9D9;margin: 10px 10px 10px 10px"></div>
pub const GREY85: Rgb = Rgb { r: 217, g: 217, b: 217 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#DBDBDB;margin: 10px 10px 10px 10px"></div>
pub const GREY86: Rgb = Rgb { r: 219, g: 219, b: 219 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#DEDEDE;margin: 10px 10px 10px 10px"></div>
pub const GREY87: Rgb = Rgb { r: 222, g: 222, b: 222 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#E0E0E0;margin: 10px 10px 10px 10px"></div>
pub const GREY88: Rgb = Rgb { r: 224, g: 224, b: 224 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#E3E3E3;margin: 10px 10px 10px 10px"></div>
pub const GREY89: Rgb = Rgb { r: 227, g: 227, b: 227 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#E5E5E5;margin: 10px 10px 10px 10px"></div>
pub const GREY90: Rgb = Rgb { r: 229, g: 229, b: 229 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#E8E8E8;margin: 10px 10px 10px 10px"></div>
pub const GREY91: Rgb = Rgb { r: 232, g: 232, b: 232 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EBEBEB;margin: 10px 10px 10px 10px"></div>
pub const GREY92: Rgb = Rgb { r: 236, g: 236, b: 236 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#EDEDED;margin: 10px 10px 10px 10px"></div>
pub const GREY93: Rgb = Rgb { r: 237, g: 237, b: 237 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F0F0F0;margin: 10px 10px 10px 10px"></div>
pub const GREY94: Rgb = Rgb { r: 240, g: 240, b: 240 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F2F2F2;margin: 10px 10px 10px 10px"></div>
pub const GREY95: Rgb = Rgb { r: 242, g: 242, b: 242 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F5F5F5;margin: 10px 10px 10px 10px"></div>
pub const GREY96: Rgb = Rgb { r: 246, g: 246, b: 246 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#F7F7F7;margin: 10px 10px 10px 10px"></div>
pub const GREY97: Rgb = Rgb { r: 248, g: 248, b: 248 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FAFAFA;margin: 10px 10px 10px 10px"></div>
pub const GREY98: Rgb = Rgb { r: 250, g: 250, b: 250 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FCFCFC;margin: 10px 10px 10px 10px"></div>
pub const GREY99: Rgb = Rgb { r: 252, g: 252, b: 252 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#FFFFFF;margin: 10px 10px 10px 10px"></div>
pub const GREY100: Rgb = Rgb { r: 255, g: 255, b: 255 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#A9A9A9;margin: 10px 10px 10px 10px"></div>
pub const DARKGREY: Rgb = Rgb { r: 169, g: 169, b: 169 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#00008B;margin: 10px 10px 10px 10px"></div>
pub const DARKBLUE: Rgb = Rgb { r: 0, g: 0, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#008B8B;margin: 10px 10px 10px 10px"></div>
pub const DARKCYAN: Rgb = Rgb { r: 0, g: 139, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B008B;margin: 10px 10px 10px 10px"></div>
pub const DARKMAGENTA: Rgb = Rgb { r: 139, g: 0, b: 139 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#8B0000;margin: 10px 10px 10px 10px"></div>
pub const DARKRED: Rgb = Rgb { r: 139, g: 0, b: 0 };
/// <div style="border:1px solid black;width:100px;height:100px;background-color:#90EE90;margin: 10px 10px 10px 10px"></div>
pub const LIGHTGREEN: Rgb = Rgb { r: 144, g: 238, b: 144 };