/// 把json配置转换成对应的struct

extern crate serde_json;
extern crate serde;

use serde_json::{Value, Map};
use serde::{Deserialize, Serialize};
use std::collections::HashMap;

/// 根据table.json文件内容，产生所有表的定义
pub fn tables_def(config: &Value) -> serde_json::Result<HashMap<String, Table>> {
    let mut result = HashMap::new();

    for (k, v) in tables_iter(config) {
        result.insert(k.to_string(), Table::new(v)?);
    }

    Ok(result)
}

/// 表的配置，包括主键，字段列表等内容
#[derive(Debug, Serialize, Deserialize)]
pub struct Table {
    pub key: Option<String>,
    pub fields: HashMap<String, Field>,
}

/// 字段定义，包括类型等内容
#[derive(Debug, Serialize, Deserialize)]
pub enum Field {
    Int,
    Double,
    String,
    Datetime,
    OneToOne(String, String),
    OneToMany(String, String),
    ManyToOne(String, String),
}

impl Table {
    fn new(json: &Value) -> serde_json::Result<Self> {
        Ok(serde_json::from_value(json.clone())?)
    }
}

// 取json对象, 取不到报错
pub fn json_object(json: &Value) -> Result<&Map<String, Value>, String> {
    // 产生所有表的全局定义
    match json {
        serde_json::Value::Object(map) => Ok(&map),
        _ => Err(format!("不是json对象，{:?}", &json))
    }
}

/// 取数据库表迭代
/// @param config: 配置文件的根
pub fn tables_iter(config: &Value) -> &Map<String, Value> {
    match &config["tables"] {
        serde_json::Value::Object(map) => &map,
        _ => panic!("tables.json的tables内容必须是json对象")
    }
}
