use crate::util::json::json_object;
use std::fs::File;
use std::str::FromStr;
use std::collections::HashMap;
use datetime::LocalDateTime;
use std::rc::Rc;
use crate::gas::entity;
use crate::search::database;
use crate::search::table::Entity;

// 代表一个产生数据的迭代
#[derive(Debug)]
struct Data {
    // 当前所有字段最新值
    values: Rc<HashMap<String, Value>>
}

// 代表一个定义的值
#[derive(Debug)]
pub enum Value {
    Int(i32),
    Double(f64),
    String(String),
    DateTime(LocalDateTime),
    // 表示一批可选值，(当前选项下标，可选值)
    Array(usize, Rc<Vec<Value>>),
    // 表示不存在
    None,
}

impl Data {
    // 根据配置产生当前值的初始内容
    fn new(define: &serde_json::Value) -> Self {
        let mut result = HashMap::new();

        for (field_name, field_define) in json_object(define).expect(&format!("期望json对象，{:?}", define)) {
            // 按from给初值
            let value = match &field_define["from"] {
                serde_json::Value::Number(x) => Value::Int(x.as_i64().expect(&format!("from只能是整数，{:?}", field_define)) as i32),
                // 字符串形式开始值，按日期处理
                serde_json::Value::String(x) => Value::DateTime(LocalDateTime::from_str(&x).expect(&format!("日期格式不合法：{}", x))),
                // 数组，取数组中第一项
                serde_json::Value::Array(array) => Value::Array(0, Rc::new(to_array(&array))),
                _ => panic!("from不支持这个数据类型, {:?}", field_define)
            };
            result.insert(field_name.to_string(), value);
        }

        Data{values: Rc::new(result)}
    }

    // 取得下一个值
    fn next(&mut self, define: &serde_json::Value) -> Rc<HashMap<String, Value>> {
        let mut result = HashMap::new();

        // 对每一个当前值，根据规则产生下一个值
        for (field_name, value) in self.values.iter() {
            // 到下一个值
            let ret = to_next(&value);
            let end = to_value(&value, &field_name, define);
            // 超出，从头开始，否则，到下一个值
            let v = if out(&ret, &end) {
                from_value(&value, &field_name, define)
            } else {
                ret
            };
            result.insert(field_name.to_string(), v);
        }

        // 把当前值转json对象
        let ret = Rc::clone(&self.values);
        // 产生下一个值
        self.values = Rc::new(result);
        ret
    }
}

impl Value {
    fn get_array_index(&self) -> usize {
        match self {
            Value::Array(i, _) => *i,
            _ => panic!("只有Array才有索引, {:?}", self)
        }
    }

    fn get_int(&self) -> i32 {
        match self {
            Value::Int(x) => *x,
            _ => panic!("只有Int才能取整数值, {:?}", self)
        }
    }

    // 转整数，可选内容取可选内容的选中项进行转换
    pub fn as_int(&self) -> i32 {
        match self {
            Value::Int(x) => *x,
            Value::Array(i, array) => array[*i].as_int(),
            _ => panic!("只有Int才能取整数值, {:?}", self)
        }
    }

    // 转浮点数，可选内容取可选内容的选中项进行转换
    pub fn as_double(&self) -> f64 {
        match self {
            Value::Double(x) => *x,
            Value::Array(i, array) => array[*i].as_double(),
            _ => panic!("只有Double才能取double值, {:?}", self)
        }
    }

    // 转整数，可选内容取可选内容的选中项进行转换
    pub fn as_string(&self) -> String {
        match self {
            Value::String(x) => x.to_string(),
            Value::Array(i, array) => array[*i].as_string(),
            _ => panic!("只有String才能取字符串值, {:?}", self)
        }
    }

    // 转日期，日期按整数表示
    pub fn as_datetime(&self) -> i32 {
        match self {
            Value::DateTime(x) => {
                // 与1970-1-1之间相差的秒数
                let zero = LocalDateTime::at(0);
                (x.to_instant().seconds() - zero.to_instant().seconds()) as i32
            },
            Value::Array(i, array) => array[*i].as_datetime(),
            _ => panic!("只有Datetime才能取日期, {:?}", self)
        }
    }
}

// 把一批json数据转换成值
fn to_array(array: &Vec<serde_json::Value>) -> Vec<Value> {
    array.iter().map(|x| {
        match x {
            serde_json::Value::Number(x) => {
                if x.is_i64() {
                    Value::Int(x.as_i64().expect(&format!("已经知道是i64，{:?}", array)) as i32)
                } else {
                    Value::Double(x.as_f64().expect(&format!("只能是f64，{:?}", array)))
                }
            },
            serde_json::Value::String(x) => Value::String(x.to_string()),
            _ => panic!("选择内容只能是字符串或者数字, {:?}", array)
        }
    }).collect()
}

// 到下一个值
fn to_next(data: &Value) -> Value {
    match data {
        Value::Int(v) => Value::Int(v+1),
        // 时间取下一个值，往后推1秒
        Value::DateTime(x) => Value::DateTime(x.add_seconds(1)),
        Value::Array(i, values) => Value::Array(i + 1, Rc::clone(values)),
        _ => panic!("不支持取下一个值, {:?}", data)
    }
}

// 看当前值是否超出结束值
fn out(data: &Value, end: &Value) -> bool {
    match end {
        // 没有结束值，永远不会超出
        Value::None => false,
        Value::Int(v) => data.get_int() > *v,
        // 是数组，看数组下标是否超出
        Value::Array(end_index, _) => {
            let data_index = data.get_array_index();
            data_index > *end_index
        },
        _ => panic!("不支持超出比较, {:?}", end)
    }
}

// 取开始值，如果是数组，开始值回到第一项
fn from_value(data: &Value, name: &str, define: &serde_json::Value) -> Value {
    match data {
        Value::Array(_, vec) => {
            Value::Array(0, Rc::clone(vec))
        },
        _ => {
            match &define[name]["from"] {
                serde_json::Value::Null => Value::None,
                serde_json::Value::Number(x) => Value::Int(x.as_i64().expect(&format!("只有整数支持to, {:?}", define)) as i32),
                _ => panic!("to不支持这个数据类型, {:?}", define)
            }
        }
    }
}

// 取结束值，没有，返回Value值的None，如果是数组，结束值为数组长度-1
fn to_value(data: &Value, name: &str, define: &serde_json::Value) -> Value {
    match data {
        Value::Array(_, vec) => Value::Array(vec.len() - 1, Rc::clone(vec)),
        _ => {
            match &define[name]["to"] {
                serde_json::Value::Null => Value::None,
                serde_json::Value::Number(x) => Value::Int(x.as_i64().expect(&format!("只有整数支持to, {:?}", define)) as i32),
                _ => panic!("to不支持这个数据类型, {:?}", define)
            }
        }
    }
}

// 创建数据，把创建对数据交给过程处理
fn proc_data<F>(data_name: &str, proc: F) where F: Fn(&String, Rc<HashMap<String, Value>>) {
    // 加载文件
    let file_name = format!("tests/{}", data_name);
    let f = File::open(&file_name).expect(&format!("文件{}不存在", file_name));
    let json = serde_json::from_reader(f).expect(&format!("文件{}不符合json结构", file_name));

    // 对每一个表，按配置加载数据
    for (table_name, table_def) in json_object(&json).expect(&format!("期望json对象，{:?}", json)) {
        let mut data = Data::new(&table_def["fields"]);

        // 获取要产生对数据个数，根据个数进行循环
        let num = table_def["nums"].as_i64().expect(&format!("必须有nums内容, {:?}", table_def));
        for _ in 1..num+1 {
            let json_data = data.next(&table_def["fields"]);
            proc(table_name, json_data);
        }
    }
}

pub fn test_data(data_name: &str) {
    proc_data(data_name, |table_name, json_data| {
        insert_test!();
    })
}

// 根据描述自动创建数据
// @param data_name: 数据描述文件的名字
pub fn create_data(data_name: &str) {
    proc_data(data_name, |table_name, json_data| {
        insert_data!();
    })
}
