use std::fmt::Debug;
use futures::TryStreamExt;
use sqlx::mssql::{MssqlConnection, MssqlRow};
use std::collections::BTreeMap;
use std::collections::HashMap;

use crate::util::json;
use crate::util::json::{Field};
use crate::test::util::Value;
use crate::update::{ClosureType, Update};
use std::marker::{Send, Sync};
use std::cmp::Ord;

// 实现表中数据加载
pub trait Entity<K> {
    // 加载一条数据库记录
    fn new(row: &MssqlRow) -> Self;
    // 产生组件值
    fn key(&self) -> K;
    // 克隆出自己的主体部分，不含关系内容
    fn clone(&self) -> Self;
    // 复制数据内容，滤过关系内容
    fn copy(key: &K, other: Self) -> ClosureType;
    // 把实体放到数据库中
    fn put(self) -> ClosureType;

    // 产生一个测试数据，用于单元测试
    fn new_test(value: &HashMap<String, Value>) -> Self;
}

// 单个表定义
#[derive(Debug)]
pub struct Table<K, T> where K: Ord, T: Entity<K> + Send + Sync {
    // 按主键索引的数据
    pub data: BTreeMap<K, T>,
}

impl<K, T> Table<K, T> where K: Ord, T: Entity<K> + Send + Sync, K: Send + Sync {
    /// 刚启动时，加载所有数据
    /// table: 某个数据库表配置文件
    /// f: 数据处理过程
    pub async fn load_data<F>(&self, db_time: &Update, name: &str, table: &json::Table, conn: &mut MssqlConnection, f: &F) -> Result<(), sqlx::Error>
    where F: Fn(ClosureType) {
        println!("加载表：{}", name);
        // 从数据库里加载数据
        let sql = get_sql(db_time, name, &table.fields);
        let mut rows = sqlx::query(&sql).fetch(conn);

        // 从数据库中读取记录，够一页数据，把数据存放到数据列表中
        while let Some(row) = rows.try_next().await? {
            // 调用数据处理过程的处理函数
            let record = T::new(&row);
            // 执行把自己添加的过程
            let proc: ClosureType = record.put();
            f(proc);
        }

        println!("加载完成，表：{}, 记录数：{}", name, self.data.len());
        Ok(())
    }

    // 后续加载。如果数据已经存在，不能直接插入，那样会破坏已经建立好的关系。
    // 返回给处理函数内容：(更新语句，更新的数据的主键，原来数据)。
    // 更新的数据主键，用于后续根据主键找到数据，建立外键关系。
    // 原来的数据，用于后续根据原来的数据内容，去掉外键关系。
    pub async fn load_next<F>(&self, db_time: &Update, name: &str, table: &json::Table, conn: &mut MssqlConnection, mut f: F) -> Result<(), sqlx::Error>
    where F: FnMut(ClosureType, K, Option<T>) {
        // 从数据库里加载数据
        let sql = get_sql(db_time, name, &table.fields);
        let mut rows = sqlx::query(&sql).fetch(conn);

        // 从数据库中读取记录，够一页数据，把数据存放到数据列表中
        while let Some(row) = rows.try_next().await? {
            // 调用数据处理过程的处理函数
            let record = T::new(&row);
            self.proc_record(record, &mut f);
        }

        Ok(())
    }

    // 清空表的内容，测试时，新的测试，要在空的数据库上进行
    pub fn clear(&mut self) {
        self.data.clear();
    }

    // 添加一条数据，用于测试过程中产生初始数据
    pub fn add_data(&mut self, data: T) {
        let key = data.key();
        self.data.insert(key, data);
    }

    // 插入一条后续测试数据
    pub fn add_test<F>(&self, record: T, mut f: F) where F: FnMut(ClosureType, K, Option<T>) {
        self.proc_record(record, &mut f);
    }

    // 处理一条数据
    fn proc_record<F>(&self, record: T, f: &mut F) where F: FnMut(ClosureType, K, Option<T>) {
        let key = record.key();
        // 看数据是否已经存在，存在，克隆旧的出来，复制新的进去
        if let Some(old) = self.data.get(&key) {
            let old_entity = old.clone();
            let update = T::copy(&key, record);
            f(update, key, Some(old_entity));

        } else {
            let update = record.put();
            f(update, key, None);
        }
    }
}

// 根据字段定义及表名，构造加载用的sql语句
fn get_sql(db_time: &Update, name: &str, fields: &HashMap<String, json::Field>) -> String {
    // 根据字段名及类型拼接select部分
    let select = fields.iter().fold("".to_string(), |acc, (field_name, field_attr)| {
        // 根据字段类型产生不同到获取字段方式
        // 一对多及一对一关系不用添加字段，返回空串
        let one = match field_attr {
            Field::Int => field_name.to_string(),
            Field::String => format!("cast({} as nvarchar) {}", field_name, field_name),
            Field::Double => format!("cast({} as float) {}", field_name, field_name),
            Field::Datetime => format!("DATEDIFF(second, '1970-01-01', {}) {}", field_name, field_name),
            // 一对多，多对一或者一对一关系，不进行数据库加载，加载完后，建立关系
            Field::OneToMany(_, _) | Field::OneToOne(_, _) | Field::ManyToOne(_, _) => "".to_string(),
        };

        // 如果不用添加，返回累积不变
        if one.is_empty() {
            acc
        } else if acc.is_empty() {
            // 第一次添加，不加逗号
            one
        } else {
            format!("{},{}", &acc, &one)
        }
    });

    let result = format!("select {} from {} where s_timestamp > {} and s_timestamp <= {}", &select, name, db_time.last_time, db_time.this_time);
    result
}
