extern crate serde_json;
extern crate lazy_static;

use crate::gas::entity;

use crate::search::table::Table;
use crate::update::{ClosureType, Update};

use std::fs::File;
use datetime::{LocalDateTime, LocalDate, LocalTime, DatePiece, Month};
use std::str::FromStr;
use std::collections::BTreeMap;
use sqlx::mssql::{MssqlConnection};
use sqlx::Connection;
use crate::util::json;
use std::collections::HashMap;

// 数据库所有表存放位置，按map存放，格式 表名:表结构
tables!();

// 系统第一次启动时，加载所有表数据
pub async fn load(db_time: &Update) -> Result<(), sqlx::Error> {
    // 加载表结构配置文件
    let f = File::open("tables.json").unwrap();
    let json: serde_json::Value = serde_json::from_reader(f).unwrap();
    let tables_def = tables_def(&json);

    // 获取连接
    let conn_str = json["conn"].as_str().expect(&format!("tables.json必须有conn指明数据库连接, {:?}", json));
    let mut conn = MssqlConnection::connect(&conn_str).await?;
    println!("连接成功");

    // 加载数据
    unsafe { 
        let f = |x: ClosureType| x();
        DATABASE.load(db_time, &tables_def, &mut conn, f).await? 
    };
    init();
    Ok(())
}

// 后续加载过程，后续加载的内容，只对新增数据处理关系
pub async fn load_next<F>(db_time: &Update, f: F) -> Result<(), sqlx::Error> where F: Fn(ClosureType) {
    // 加载表结构配置文件
    let file = File::open("tables.json").unwrap();
    let json: serde_json::Value = serde_json::from_reader(file).unwrap();
    let tables_def = tables_def(&json);

    // 获取连接
    let conn_str = json["conn"].as_str().expect(&format!("tables.json必须有conn指明数据库连接, {:?}", json));
    let mut conn = MssqlConnection::connect(&conn_str).await?;

    // 加载数据
    DataBase::load_next(db_time, &tables_def, &mut conn, f).await?;
    Ok(())
}

// 数据加载完成后的初始化过程，包括建立主键，建立关系等等
pub fn init() {
    // 一对多关系
    DataBase::create_relation(|x| x());
}

// 清空数据库
pub fn clear() {
    unsafe {
        DATABASE.clear();
    }
}

// 查询过程
// @param name: 查询名
// @param params: json格式的查询参数
pub fn search(name: &str, params: &serde_json::Value) -> Result<String, String> {
    // 不分页，这两个变量不应该在程序中出现，暂时放上去，以便编译通过
    let page_start = 0 as usize;
    let page_end = 0 as usize;
    let page_total = false;
    
    path!()
}

// 查询总和
pub fn search_total(name: &str, params: &serde_json::Value) -> Result<String, String> {
    let page_start = 0 as usize;
    let page_end = 0 as usize;

    // 设置查询total
    let page_total = true;

    path!()
}

// 分页查询
pub fn search_page(name: &str, params: &serde_json::Value, page_index: usize, page_size: usize) -> Result<String, String> {
    // 计算出迭代的范围值，自动传入path中
    let page_start = (page_index - 1) * page_size;
    let page_end = page_start + page_size;

    // 设置不按分页查询
    let page_total = false;

    path!()
}

// 选择过程
fn select(data: Vec<Vec<(&str, String)>>) -> String {
    let mut result = String::from("[");
    let mut is_first = true;

    // 对结果集中对象进行遍历，调用每一个转换过程，产生最后对json串
    for item in data {
        // 如果不是第一项，加逗号分割
        if is_first {
            is_first = false;
        } else {
            result.push(',');
        }

        let mut item_first = true;
        result.push('{');
        // 对每一条记录进行处理
        for (k, v) in item {
            // 如果不是第一项，加逗号
            if item_first {
                item_first = false;
            } else {
                result.push(',');
            }
            result.push_str(&k);
            result.push(':');
            result.push_str(&v);
        }
        result.push('}');
    }
    result.push(']');
    result
}

// 获取连接
pub async fn get_connect() -> Result<MssqlConnection, sqlx::Error> {
    // 加载表结构配置文件
    let f = File::open("tables.json").unwrap();
    let json: serde_json::Value = serde_json::from_reader(f).unwrap();

    // 获取连接
    let conn_str = json["conn"].as_str().expect(&format!("tables.json必须有conn指明数据库连接, {:?}", json));
    let conn = MssqlConnection::connect(&conn_str).await?;
    Ok(conn)
}

// 获得所有表的定义内容
fn tables_def(config: &serde_json::Value) -> HashMap<String, json::Table> {
    match json::tables_def(config) {
        Ok(v) => v,
        Err(err) => panic!("tables.json文件有错误，{:?}", err)
    }
}

// 根据月份，得到月份的枚举表示
fn get_month_enum(m: i8) -> Month {
    match m {
        1 => Month::January,
        2 => Month::February,
        3 => Month::March,
        4 => Month::April,
        5 => Month::May,
        6 => Month::June,
        7 => Month::July,
        8 => Month::August,
        9 => Month::September,
        10 => Month::October,
        11 => Month::November,
        12 => Month::December,
        _ => panic!("月份错误, {}", m)
    }
}

/// 字符串转日期
fn datetime(x: &str) -> i32 {
    let date = LocalDateTime::from_str(x).expect(&format!("日期格式不合法：{}", x));
    // 转换成与1970年1月1日的差
    seconds(&date)
}

/// 获取日期的年
fn year(x: i32) -> i64 {
    let date = LocalDateTime::at(x as i64);
    date.date().year()
}

// 产生某天开始时间
fn date_start(y: i64, m: i8, d: i8) -> i32 {
    let date = LocalDateTime::new(
        LocalDate::ymd(y, get_month_enum(m), d).expect(&format!("年月日不合法：{}, {}, {}", y, m, d)), 
        LocalTime::hms(0, 0, 0).expect("search, 时间不可能出错")
    );
    seconds(&date)
} 

// 产生某天结束时间
fn date_end(y: i64, m: i8, d: i8) -> i32 {
    let date = LocalDateTime::new(
        LocalDate::ymd(y, get_month_enum(m), d).expect(&format!("年月日不合法：{}, {}, {}", y, m, d)), 
        LocalTime::hms(23, 59, 59).expect("search, 时间不可能出错")
    );
    seconds(&date)
}

// 把日期转换成与1970年1月1日之间秒的差值
fn seconds(date: &LocalDateTime) -> i32 {
    // 转换成与1970年1月1日的差
    let start = LocalDateTime::at(0);
    (date.to_instant().seconds() - start.to_instant().seconds()) as i32
}