#![feature(const_btree_new)]
//#![feature(async_closure)]

#[macro_use] 
extern crate rocket;

extern crate serde_json;
extern crate mylib;

use std::{thread, time};

use mylib::search::database;
use mylib::update::Update;
//use std::sync::RwLock;
use async_std::task;

#[rocket::main]
async fn main() -> Result<(), String> {
    //let lock = RwLock::new(0);

    // 数据库时钟，每次取值，将获得数据库时钟数据
    let mut db_time = Update::new();
    // 加载数据库中数据
    database::load(&db_time).await.map_err(|x| x.to_string())?;

    // 启动数据读取线程
    thread::spawn(move || -> Result<(), sqlx::Error> {
        // 从更新库，获得一个更新过程。循环调用更新过程，完成更新。
        loop {
            // 获得时钟，时钟没有变，数据库状态没变，不处理
            db_time = db_time.next();
            if db_time.this_time > db_time.last_time {
                task::block_on(async {
                    database::load_next(&db_time, |x| {
                        // 锁定数据库，执行更新过程
                        //let _w = lock.write().unwrap();
                        x();    
                    }).await.expect("加载数据出错");
                });
            }

            // 间歇1秒
            let ten_millis = time::Duration::from_millis(1000);
            thread::sleep(ten_millis);
        }
    });

    // 启动服务
    rocket().launch().await.map_err(|x| x.to_string())?;
    Ok(())
}

#[post("/<path>", data = "<conds>")]
fn index(path: String, conds: String) -> Result<String, String> {
    let params: serde_json::Value = serde_json::from_str(&conds).unwrap();
    // 根据path查询名执行查询
    database::search(&path, &params)
}

/// 求分页查询的总和
#[post("/<path>/n", data = "<conds>")]
fn total(path: String, conds: String) -> Result<String, String>  {
    let params: serde_json::Value = serde_json::from_str(&conds).unwrap();
    // 根据path查询名执行查询
    database::search_total(&path, &params)
}

/// 查询某页数据
#[post("/<path>/<page_index>/<page_size>", data = "<conds>")]
fn page(path: String, page_size: usize, page_index: usize, conds: String) -> Result<String, String> {
    let params: serde_json::Value = serde_json::from_str(&conds).unwrap();
    // 根据path查询名执行查询
    database::search_page(&path, &params, page_index, page_size)
}

fn rocket() -> rocket::Rocket<rocket::Build> {
    rocket::build().mount("/", routes![index, total, page])
}

use std::sync::Once;
use mylib::test::util::create_data;

static INIT: Once = Once::new();

pub fn initialize() {
    INIT.call_once(|| {
        create_data("one.data.json");
        // 加载并创建关系
        database::init();
    });
}

#[cfg(test)]
mod test {
    use super::rocket;
    use super::*;
    use rocket::local::blocking::Client;
    use rocket::http::Status;
    use mylib::test::util::test_data;
    use insta::assert_snapshot;

    #[test]
    fn hello_world() {
        initialize();

        let client = Client::tracked(rocket()).expect("valid rocket instance");
        let str = "{\"address\":\"西安软件园\", \"start\":\"2021-01-01T00:00:00\", \"end\":\"2021-12-31T23:59:59\"}";
        let response = client.post("/monthgas/1/5").body(str).dispatch();
        assert_eq!(response.status(), Status::Ok);
        assert_snapshot!("test1", response.into_string().unwrap());
    }

    // 测试获取总数
    #[test]
    fn count_test() {
        initialize();

        let client = Client::tracked(rocket()).expect("valid rocket instance");
        let str = "{\"address\":\"西安软件园\", \"start\":\"2021-01-01T00:00:00\", \"end\":\"2021-12-31T23:59:59\"}";
        let response = client.post("/monthgas/n").body(str).dispatch();
        assert_eq!(response.status(), Status::Ok);
        assert_snapshot!("count", response.into_string().unwrap());
    }

    // 测试a[f.sum()>0]的情况
    #[test]
    fn test_agg_filter() {
        initialize();

        let client = Client::tracked(rocket()).expect("valid rocket instance");
        let str = "{}";
        let response = client.post("/agg_filter").body(str).dispatch();
        assert_eq!(response.status(), Status::Ok);
        assert_snapshot!("test_agg_filter", response.into_string().unwrap());
    }

    #[test]
    // 测试多对一处理
    fn test_many_to_one() {
        initialize();

        let client = Client::tracked(rocket()).expect("valid rocket instance");
        let str = "{}";
        let response = client.post("/many_to_one").body(str).dispatch();
        assert_eq!(response.status(), Status::Ok);
        assert_snapshot!("many_to_one", response.into_string().unwrap());
    }

    // 测试修改过程
    #[test]
    #[ignore]
    fn test_update() {
        initialize();

        (1..100).for_each(|_| {
            test_data("update.json");
        });
    }

    #[test]
    #[ignore]
    fn test_update_search() {
        initialize();

        let client = Client::tracked(rocket()).expect("valid rocket instance");
        let str = "{\"address\":\"西安软件园\", \"start\":\"2021-01-01T00:00:00\", \"end\":\"2021-12-31T23:59:59\"}";
        (1..10000).for_each(|_| {
            let response = client.post("/monthgas/1/20").body(str).dispatch();
            assert_eq!(response.status(), Status::Ok);
            assert_snapshot!("test_update", response.into_string().unwrap());
        })
    }
}

/* 
impl Iterator for A {
    type Item = (i32, i32);
    fn next(&mut self) -> Option<Self::Item> {
        let iter = |i, j| {
            if j < i {
                iter(i, j+1)
            } else if i < n {
                iter(i+1, 0)
            } else {
                None
            }
        }
    
        Some((0, 0))
    }
}

fn enum_ij(n: i32) -> Box<dyn Iterator<Item = (i32, i32)>>{
    let iter = |i, j| {
        if j < i {
            iter(i, j+1)
        } else if i < n {
            iter(i+1, 0)
        } else {
            None
        }
    }


    return Box::new(A{})
}*/

/*
fn main() {
    loop {
        println!("请输入命令，l加载，q退出");
        let mut line = String::new();
        // 读取要执行的查询
        io::stdin().read_line(&mut line).unwrap();
        line = line.replace("\n", "");
        // 如果是quit，退出
        if line == "q" {
            break;
        } else if line == "l" {
            // 加载数据库
            database::load();
        } else {
            let params: Value = serde_json::from_str("{\"start\":\"2021-09-02T00:00:00\", \"end\":\"2021-09-02T23:59:59\"}").unwrap();
            let start = Instant::now();
            // 加载文件执行
            let result = database::search(&line, &params);
            match result {
                Ok(v) => {
                    let t = start.elapsed();
                    println!("查询结果为：{:?}毫秒, {:?}微妙, {:?}纳秒", t.as_millis(), t.as_micros(), t.as_nanos());
                    println!("{}", v.to_string());
                    println!("请继续输入查询脚本名，q退出");        
                },
                Err(e) => {
                    println!("{}，请继续", e);
                }
            }
        }
    }
}
*/