use proc_macro::{Delimiter};
use crate::expression::{Expression, QueryType, Select};
use crate::base::{Parser, map, one_or_more, zero_or_more, zero_or_one, is, set_var, has_var};
use crate::token::Tokens;

// 静态符号表
static P: [&str; 1] = ["p"];
static LET: [&str; 1] = ["let"];
static DB: [&str; 1] = ["db"];
static TOTAL: [&str; 1] = ["total"];
static G: [&str; 1] = ["g"];
static IF: [&str; 1] = ["if"];
static ELSE: [&str; 1] = ["else"];
static DOT: [&str; 1] = ["."];
static COMMA: [&str; 1] = [","];
static COLON: [&str; 1] = [":"];
static ASSIGN: [&str; 1] = ["="];
static LOGIC: [&str; 2] = ["&&", "||"];
static COMPARE: [&str; 6] = ["==", "!=", ">", ">=", "<", "<="];
static MATH: [&str; 2] = ["+", "-"];
static MUL: [&str; 3] = ["*", "/", "%"];

/// 查询语句解析
/// @param tokens: 单词流
/// @param del: 解析过程中，获得的已经定义的函数名，以便后续解析过程可以用这个函数名
pub fn parse(tokens: &mut Tokens, del: &mut Vec<String>) -> Result<Expression, String> {
    // 从程序开始解析
    let parser = program();
    let proc = (parser.run)(tokens, del)?.0.expect("program.parse, 不会没有内容");
    Ok(proc)
}

/// 程序:=语句+
fn program() -> Parser<Expression> {
    map(|exps| Expression::Program(exps), one_or_more(Box::new(|| statement())))
}

/// 语句:= 参数说明 | let 函数声明 | 表达式
fn statement() -> Parser<Expression> {
    func() | Box::new(|| param_def()) | Box::new(|| exp())
}

/// 参数定义:=p {(字段名: 字段类型,)*}
fn param_def() -> Parser<Expression> {
    let result = |exps| Expression::Params(exps);
    
    spec_ident(&P) >>
    Box::new(move || map(result, group(Delimiter::Brace, array(|| var_def()))))
}

/// 变量定义:=变量名:变量类型
fn var_def() -> Parser<Expression> {
    let exp = |var: syn::Ident| move |ty: syn::Ident| 
        Expression::Var(var.to_string(), ty.to_string());

    (map(exp, ident().err("必须是变量名")) << Box::new(|| oper(&COLON).err("必须是冒号"))) + 
    Box::new(|| ident().err("必须是变量类型"))
}

/// 函数声明 := let 函数名 = 表达式
fn func() -> Parser<Expression> {
    let result = |name| |exp| Expression::Func(name, Box::new(exp));

    map(result, key(&LET) >> Box::new(|| set_var(ident().err("必须是函数名"))) << 
    Box::new(|| oper(&ASSIGN).err("必须是等号"))) + Box::new(|| exp())
}

/// 表达式先按逻辑表达式考虑
fn exp() -> Parser<Expression> {
    condition()
}

/// 条件表达式 := 逻辑表达式 | if 逻辑表达式 {程序} （elif 逻辑表达式 {程序})* else {程序}
fn condition() -> Parser<Expression> {
    if_exp() | Box::new(|| logic())
}

/// if表达式:=if 逻辑表达式 {程序} （elif 逻辑表达式 {程序})* else {程序}
fn if_exp() -> Parser<Expression> {
    let result = |if_cond| |if_exp| |else_exp|
        Expression::Cond(Box::new(if_cond), Box::new(if_exp), Box::new(else_exp));

    map(result, key(&IF) >> Box::new(|| logic())) + 
    Box::new(|| group(Delimiter::Brace, program()).err("必须是大括号")) +
    Box::new(||
        key(&ELSE).err("必须是else") >> 
        Box::new(|| group(Delimiter::Brace, program()).err("必须是打括号"))
    )
}

/// 逻辑表达式:=比较表达式 (&&|(||) 比较表达式)* | !逻辑表达式
fn logic() -> Parser<Expression> {
    binary(&LOGIC, || compare())
}

/// 比较表达式=算数表达式((==, !=, >, >=, <, <=)算数表达式)?
fn compare() -> Parser<Expression> {
    let result = |left| |right| {
        match right {
            None => left,
            Some((oper, right)) => Expression::Binary(oper, Box::new(left), Box::new(right))
        }
    };

    // 算数表达式 比较运算符 算数表达式
    map(result, math()) + 
    Box::new(|| zero_or_one(Box::new(|| {
        let oper_one = |oper| |right| (oper, right);
        map(oper_one, oper(&COMPARE)) + Box::new(|| math())
    })))
}

/// 算数表达式:=乘法表达式((+|-)乘法表达式)*
fn math() -> Parser<Expression> {
    binary(&MATH, || mul())
}

/// 乘法表达式:=算子((*|/|%)算子)*
fn mul() -> Parser<Expression> {
    binary(&MUL, || atom())
}

/// 算子:=常数 | 函数调用 | 查询语句 | 变量
fn atom() -> Parser<Expression> {
    // (表达式)
    paren_exp() | 
    // 常数
    Box::new(|| lit()) |
    // rust函数调用 
    Box::new(|| rust_call()) | 
    // 参数调用
    Box::new(|| param_value()) |
    // 数据库查询路径
    Box::new(|| db()) |
    // 外部变量
    Box::new(|| var()) |
    // 聚集函数，纯聚集函数按无名路径看待
    Box::new(|| query_agg()) |
    // 子查询或者变量查询，在执行期间根据上下文决定
    Box::new(|| sub_or_var(false))
}

/// ()包括的表达式
fn paren_exp() -> Parser<Expression> {
    map(|result| Expression::Expr(Box::new(result)), group(Delimiter::Parenthesis, exp()))
}

/// 常数
fn lit() -> Parser<Expression> {
    is(|token| token.is_lit(), |token, _del| {
        let lit = token.to_lit();
        Expression::Lit(lit.clone())
    })
}

/// 参数取值:=p.参数名
fn param_value() -> Parser<Expression> {
    let result = |name: syn::Ident| Expression::ParamValue(name.to_string());
    map(result, spec_ident(&P) >> 
    Box::new(|| oper(&DOT).err("必须是‘.'")) >> Box::new(|| ident().err("必须是标识符")))
}

/// rust函数:=g.函数名(函数参数)
fn rust_call() -> Parser<Expression> {
    let result = |name| |params| Expression::Call(name, params);
    map(result, spec_ident(&G) >> 
    Box::new(|| oper(&DOT).err("必须是'.'")) >> Box::new(|| ident().err("必须是标识符"))) + 
    Box::new(|| func_param())
}

/// 外部变量
fn var() -> Parser<Expression> {
    is(|token| token.starts_with("page_"), |token, _del| {
        let ident = token.to_ident();
        Expression::Ident(ident.clone())
    })
}

/// 纯聚集函数，按无名查询路径看待
fn query_agg() -> Parser<Expression> {
    let result = |select| Expression::Query(QueryType::Sub, None, Vec::new(), Box::new(select));
    map(result, select_agg())
}

/// 顶部数据查询路径:=db.path路径
fn db() -> Parser<Expression> {
    spec_ident(&DB) >> Box::new(|| oper(&DOT).err("必须是'.'")) >> Box::new(|| sub_or_var(true))
}

/// 变量查询
/// is_db: 是否全局数据库查询
fn sub_or_var(is_db: bool) -> Parser<Expression> {
    let result = move |(ident, has)| move |(cond, select)| {
        // 如果上下文有名字，按变量查询，否则，按子查询
        let ty = if is_db {
            QueryType::Database
        } else if has {
            QueryType::Var
        } else {
            QueryType::Sub
        };
        Expression::Query(ty, Some(ident), cond, Box::new(select))
    };
    map(result, has_var(ident())) + Box::new(|| query())
}

/// query:= [条件]* select?
fn query() -> Parser<(Vec<Expression>, Select)> {
    let result = |cond| |select| (cond, select);
    map(result, cond()) + Box::new(|| select())
}

/// 过滤条件:=([逻辑表达式|范围表达式])*
fn cond() -> Parser<Vec<Expression>> {
    zero_or_more(Box::new(|| {
        group(Delimiter::Bracket, range() | Box::new(|| logic()))
    }))
}

/// 范围处理
fn range() -> Parser<Expression> {
    Parser::new(Box::new(|tokens, _del| {
        // 可以按范围处理，返回rust的范围表达式
        if let Ok(range) = syn::parse_str::<syn::ExprRange>(tokens.source.as_str()) {
            Ok((Some(Expression::Range(range.clone())), Vec::new()))
        } else {
            Ok((None, Vec::new()))
        }
    }))
}

/// 选择字段:= .名字 | .函数调用 | .(名字，名字) | .total(名字，名字)
/// 如果是获得迭代本身的聚集函数内容，加上total标识符，进行识别。
/// 返回：选择内容，选择内容可以是属性，也可以是函数调用，也可以是一批内容
fn select() -> Parser<Select> {
    // 有.开始的选择过程
    let has_select = oper(&DOT) >> Box::new(||
        (total() | Box::new(|| select_agg()) | Box::new(|| select_one()) | Box::new(|| select_total(false)) ));
    // 没有选择的情况下，返回无选择过程
    has_select | Box::new(|| no_select())
}

/// 单选择，把ident转select
fn select_one() -> Parser<Select> {
    let result = |ident| Select::Field(ident);
    map(result, ident())
}

/// 集合本身聚集
fn total() -> Parser<Select> {
    spec_ident(&TOTAL) >> Box::new(|| select_total(true))
}

/// 选择一批聚集函数值
fn select_total(ty: bool) -> Parser<Select> {
    let result = move |exps: Vec<Expression>| {
        let selects_exp = exps.into_iter().map(|v| {
            // 获取属性名及表达式
            let (ident, e) = v.to_attr();
            (ident, *e)
        }).collect::<Vec<(syn::Ident, Expression)>>();
        Select::Selects(ty, selects_exp)
    };
    map(result, group(Delimiter::Parenthesis, array(|| attr())))
}

/// 取到一项选择内容及别名
fn attr() -> Parser<Expression> {
    let result = |exp| |ident| Expression::Attr(ident, Box::new(exp));
    map(result, exp()) + Box::new(|| ident())
}

/// 聚集函数，把函数参数当表达式，函数本身独立出来返回
/// 返回：表达式及聚集函数名，聚集函数可能没有参数，所以表达式可能为空，可以是普通函数，所以聚集函数名，也可以为空
fn select_agg() -> Parser<Select> {
    let result = |ident| |mut exps: Vec<Expression>| {
        let first = exps.pop();
        Select::Agg(ident, first)
    };
    map(result, ident()) + Box::new(|| func_param())
}

/// 函数参数:=(一批逗号分割的表达式?)
fn func_param() -> Parser<Vec<Expression>> {
    let result = |value| {
        match value {
            None => Vec::new(),
            Some(x) => x
        }
    };
    map(result, group(Delimiter::Parenthesis, zero_or_one(Box::new(|| array(|| exp())))))
}

/// 无选择内容
fn no_select() -> Parser<Select> {
    Parser::new(Box::new(|_tokens, _del| {
        Ok((Some(Select::Selects(false, Vec::new())), Vec::new()))
    }))
}

/// 解决这种文法，表达式 := 数字 ((+|-)数字)*
/// - p: 产生解析器的过程。
/// - opers: 需要满足的二元操作符
fn binary<P: 'static>(opers: &'static [&str], p: P) -> Parser<Expression> where P: Fn() -> Parser<Expression> + Copy {
    let result = |first| |exps: Vec<Expression>| {
        exps.into_iter().fold(first, |acc, right| {
            let (ident, exp) = right.to_oper_one();
            Expression::Binary(ident, Box::new(acc), Box::new(exp))
        })
    };
    // 先用parser处理，然后按多个(操作符，parser)处理
    map(result, p()) + 
    Box::new(move || zero_or_more(Box::new(move || {
        let one_exp = |oper| |exp| Expression::OperOne(oper, Box::new(exp));
        map(one_exp, oper(opers)) + Box::new(move || p())
    })))
}

/// 解决这种文件，函数参数 := (表达式 (,表达式)*)?，分隔符先假定为逗号
/// - p: 产生需要的解析器
/// - f: 把一批语法节点转换成一个最终节点
fn array<T: 'static, P: 'static>(p: P) -> Parser<Vec<T>> where P: Fn() -> Parser<T> + Copy {
    let result = |first| |mut exps: Vec<T>| {
        exps.insert(0, first);
        exps
    };
    // 先用parser处理，然后在按多个(,parser)处理
    map(result, p()) +
    Box::new(move || zero_or_more(Box::new(move || 
        oper(&COMMA) >> Box::new(move || p())
    )))
}

/// 取标识符
fn ident() -> Parser<syn::Ident> {
    is(|token| token.is_ident(), |token, _del| token.to_ident())
}

/// 取给定标识符
fn spec_ident(idents: &'static [&str]) -> Parser<syn::Ident> {
    is(move |token| token.is_spec_ident(idents), |token, _del| token.to_ident())
}

/// 看是否标识符，是对情况下，转换成字符串。用于let这种无法转换成标识符对情况
fn key(idents: &'static [&str]) -> Parser<String> {
    is(move |token| token.is_spec_ident(idents), |token, _del| token.to_string())
}

/// 取操作符
fn oper(opers: &'static [&str]) -> Parser<String> {
    is(move |token| token.is_oper(opers), |token, _del| token.to_string())
}

/// 取group内容
fn group<T: 'static>(delimiter: proc_macro::Delimiter, parser: Parser<T>) -> Parser<T> {
    is(move |token| token.is_group(delimiter), move |token, del| {
        // 把token转单词流，让parser进行解析，返回解析结果
        let mut tokens = token.to_tokens();
        let (result, _) = (parser.run)(&mut tokens, del).expect("解析错误");
        result.expect(&format!("group一定有内容: {}", tokens.to_string()))
    })
}

/*
/// 查询语句解析
/// @param tokens: 单词流
/// @param del: 解析过程中，获得的已经定义的函数名，以便后续解析过程可以用这个函数名
pub fn parse(tokens: &mut Tokens, del: &mut Vec<String>) -> Result<Expression, String> {
    program(tokens, del)
}

/// 程序:=语句+
fn program(tokens: &mut Tokens, del: &mut Vec<String>) -> Result<Expression, String> {
    let mut result = Vec::new();
    loop {
        let statement = statement(tokens, del)?;
        match statement {
            // 没有语句了，返回
            Expression::None => break,
            // 其它，把语句加入到程序中
            _ => result.push(statement),
        }
    }
    Ok(Expression::Program(result))
}

/// 语句:= 参数说明 | let 函数声明 | 表达式
/// @param tokens: 单词流
/// @param del: 当前已经定义的函数名
fn statement(tokens: &mut Tokens, del: &mut Vec<String>) -> Result<Expression, String> {
    let token_op = tokens.next();
    match token_op {
        Some(token) => {
            // 如果是let，调用函数声明过程
            if is_key(&token, "let") {
                func(tokens, del)
            } else if is_key(&token, "p") {
                // 参数声明处理，获得参数声明
                param_def(tokens)
            } else {
                // 单词回退，调用表达式过程
                tokens.tokens.push(token);
                exp(tokens, del)
            }
        },
        // 没有读到单词，返回空
        None => Ok(Expression::None),
    }
}

/// 参数定义:={(字段名: 字段类型,)*}
fn param_def(tokens: &mut Tokens) -> Result<Expression, String> {
    // 取大括号中内容，进行解析
    let stream = get_group(tokens, Delimiter::Brace)?;
    let mut tokens = Tokens::new(stream);
    // 调用0或多的处理过程
    let attr_exp = array(&mut tokens, |tokens| {
        // 获取变量名
        let var_name = get_ident(tokens)?;
        // 读掉冒号
        get_spec_puncts(tokens, &vec![":"]).expect("必修有冒号");
        // 获取变量类型
        let var_type = get_ident(tokens)?;
        Ok(Expression::Var(var_name, var_type))
    })?;
    Ok(Expression::Params(attr_exp))
}

/// 函数声明 := 函数名 = 表达式
/// @param del: 保存已经定义的函数名
/// @param param: 参数声明，取参数值时，要根据参数声明决定怎么进行参数转换
fn func(tokens: &mut Tokens, del: &mut Vec<String>) -> Result<Expression, String> {
    // 得到函数名
    let name_str = get_ident(tokens)?;
    let name: syn::Ident = syn::parse_str(&name_str).expect(&format!("名称不合法, {}", name_str));
    // 把函数名存放到函数名列表中
    del.push(name_str.to_string());
    // 必须是“=”
    get_spec_puncts(tokens, &vec!["="]).expect(&format!("必须是="));
    // 调用表达式
    let exp = exp(tokens, del)?;
    // 返回函数声明
    Ok(Expression::Func(name, Box::new(exp)))
}

/// 表达式先按逻辑表达式考虑
fn exp(tokens: &mut Tokens, del: &mut Vec<String>) -> Result<Expression, String> {
    condition(tokens, del)
}

/// 条件表达式 := 逻辑表达式 | if 逻辑表达式 {程序} （elif 逻辑表达式 {程序})* else {程序}
fn condition(tokens: &mut Tokens, del: &mut Vec<String>) -> Result<Expression, String> {
    // 是if，按if过程处理
    if let Ok(_) = get_spec_ident(tokens, &vec!["if"]) {
        // 取if条件部分
        let if_cond = logic(tokens, del)?;
        // 取if的内容
        let stream = get_group(tokens, Delimiter::Brace)?;
        let mut if_tokens = Tokens::new(stream);
        let if_exp = program(&mut if_tokens, del)?;
        // 取else部分
        get_spec_ident(tokens, &vec!["else"])?;
        let stream = get_group(tokens, Delimiter::Brace)?;
        let mut else_tokens = Tokens::new(stream);
        let else_exp = program(&mut else_tokens, del)?;
        // 返回if表达式语法树
        Ok(Expression::Cond(Box::new(if_cond), Box::new(if_exp), Box::new(else_exp)))
    } else {
        // 按普通逻辑表达式处理
        logic(tokens, del)
    }
}

/// 逻辑表达式:=比较表达式 (&&|(||) 比较表达式)* | !逻辑表达式
fn logic(tokens: &mut Tokens, del: &Vec<String>) -> Result<Expression, String> {
    binary(tokens, &vec!["&&", "||"], |tokens| compare(tokens, del))
}

/// 比较表达式=算数表达式((==, !=, >, >=, <, <=)算数表达式)?
fn compare(tokens: &mut Tokens, del: &Vec<String>) -> Result<Expression, String> {
    let left = math(tokens, del)?;
    let ident = get_spec_puncts(tokens, &vec!["==", "!=", ">", ">=", "<", "<="]);
    let result = match ident {
        // 不是比较运算符，直接返回
        None => Ok(left),
        Some(punct) => {
            let right = math(tokens, del)?;
            Ok(Expression::Binary(punct, Box::new(left), Box::new(right)))
        }
    };
    result
}

/// 算数表达式:=乘法表达式((+|-)乘法表达式)*
fn math(tokens: &mut Tokens, del: &Vec<String>) -> Result<Expression, String> {
    binary(tokens, &vec!["+", "-"], |tokens| mul(tokens, del))
}

/// 乘法表达式:=算子((*|/|%)算子)*
fn mul(tokens: &mut Tokens, del: &Vec<String>) -> Result<Expression, String> {
    binary(tokens, &vec!["*", "/", "%"], |tokens| atom(tokens, del))
}

/// 算子:=常数 | 函数调用 | 查询语句 | 变量
/// @param del: 已经定义的函数名，为了判断调用时是函数，还是查询
fn atom(tokens: &mut Tokens, del: &Vec<String>) -> Result<Expression, String> {
    // 是()
    if let Ok(stream) = get_group(tokens, Delimiter::Parenthesis) {
        let mut tokens = Tokens::new(stream);
        let exp = logic(&mut tokens, del)?;
        return Ok(Expression::Expr(Box::new(exp)))
    }
    
    // 取函数名
    let token = tokens.next().expect("表达式必须有单词");
    match token {
        // 标识符，可能是：函数调用，查询语句，参数取值，变量
        TokenTree::Ident(_) => {
            let name = to_ident(&token)?;
            let name_str = name.to_string();

            // 如果名称在已经定义的函数列表中
            if del.contains(&name.to_string()) {
                // 如果是小括号，按函数调用处理
                if let Ok(stream) = get_group(tokens, Delimiter::Parenthesis) {
                    let mut tokens = Tokens::new(stream);
                    let (exp_op, _) = call(&mut tokens, name, del)?;
                    Ok(exp_op.expect("必须有表达式内容"))
                }
                // 不是函数调用，按查询路径处理，普通变量也支持查询路径的处理 
                else {
                    query(tokens, del, name, QueryType::Var)
                }
            } else if name_str == "p" {
                // p开始是参数取值
                param_value(tokens)
            } else if name_str == "g" {
                // g开始是调用rust定义的全局函数
                rust_call(tokens, del)
            } else if name_str.starts_with("page_") {
                // 以page_开始，是分页的变量，按普通标识符处理
                Ok(Expression::Ident(name))
            }
            // db表示取全局数据库的内容
            else if name_str == "db" {
                // 读掉“."
                get_spec_puncts(tokens, &vec!["."]).expect("db后面必须是.");
                // 读出表名
                let name_str = get_ident(tokens)?;
                let name: syn::Ident = syn::parse_str(&name_str).expect("变量名不合法");
                query(tokens, del, name, QueryType::Database)
            }
            else {
                // 否则，按查询语句处理
                query(tokens, del, name, QueryType::Sub)
            }
        },
        // 常数，对常数进行解析，存放到表达式的常数中
        TokenTree::Literal(_) => {
            let lit: syn::Lit = syn::parse(TokenStream::from(token)).expect("常数解析错误");
            Ok(Expression::Lit(lit))
        },
        _ => Err(format!("表达式只能是一个函数调用或者常数, {:?}", token.to_string())),
    }
}

/// 参数取值:=p.参数名
fn param_value(tokens: &mut Tokens) -> Result<Expression, String> {
    get_spec_puncts(tokens, &vec!["."]).expect("参数取值必须是'.'，不能是空");
    // 获取参数名
    let name = get_ident(tokens)?;
    Ok(Expression::ParamValue(name))
}

/// 调用rust定义的函数
fn rust_call(tokens: &mut Tokens, del: &Vec<String>) -> Result<Expression, String> {
    get_spec_puncts(tokens, &vec!["."]).expect("rust函数调用必须是'.'，不能是空");
    // 获取函数名
    let name = get_ident(tokens)?;
    let name: syn::Ident = syn::parse_str(&name).expect(&format!("名称不合法，{}", name));
    // 获取函数参数
    let stream = get_group(tokens, Delimiter::Parenthesis)?;
    let mut tokens = Tokens::new(stream);
    // 产生函数调用节点
    let (exp_op, _) = call(&mut tokens, name, del)?;
    Ok(exp_op.expect("必须有表达式内容"))
}

/// 函数调用，如果是聚集函数，把函数参数当表达式，函数本身独立出来返回
/// 返回：表达式及聚集函数名，聚集函数可能没有参数，所以表达式可能为空，可以是普通函数，所以聚集函数名，也可以为空
fn call(tokens: &mut Tokens, name: syn::Ident, del: &Vec<String>) -> Result<(Option<Expression>, Option<syn::Ident>), String> {
    // 获取函数参数
    let mut params = array(tokens, |tokens| {
        logic(tokens, del)
    })?;

    // 是否聚集函数，先硬判断，以后可以根据语法情况进行处理
    match name.to_string().as_str() {
        // 聚集函数，把函数参数当第一项及函数名返回
        "sum" | "count" => {
            // 有参数，取参数第一项，否则取空
            match params.pop() {
                None => Ok((None, Some(name))),
                Some(first) => Ok((Some(first), Some(name))),
            }
        },
        // 非聚集函数，产生函数调用表达式
        _ => {
            Ok((Some(Expression::Call(name, params)), None))
        }
    }
}

/// query:=表名([条件])?(.字段名)?
/// ident: 查询的名字
/// q_type: 查询类型，数据库，变量，子查询
fn query(tokens: &mut Tokens, del: &Vec<String>, ident: syn::Ident, q_type: QueryType) -> Result<Expression, String> {
    // 如果是函数调用，不是纯粹标识符，按选择内容为聚集函数处理
    match get_group(tokens, Delimiter::Parenthesis) {
        Ok(stream) => {
            // 返回聚集函数调用的选择
            let (exp, name) = call(&mut Tokens::new(stream), ident, del)?;
            let select = Select::Agg(name.expect("聚集函数必须有名字"), exp);
            Ok(Expression::Query(q_type, None, Vec::new(), Box::new(select)))
        },
        Err(_) => {
            // 获得条件
            let cond = cond(tokens, del)?;
            // 获得选择内容，包括选择内容及聚集函数，两者有可能为空
            let select = select(tokens, del)?;
            // 返回查询
            Ok(Expression::Query(q_type, Some(ident), cond, Box::new(select)))            
        }
    }
}

/// 条件表达式:=([逻辑表达式|范围表达式])*
fn cond(tokens: &mut Tokens, del: &Vec<String>) -> Result<Vec<Expression>, String> {
    let mut result = Vec::new();

    while let Ok(stream) = get_group(tokens, Delimiter::Bracket) {
        // 范围处理过程，让rust做解析
        let range_op = syn::parse::<syn::ExprRange>(stream.clone());
        let exp = match range_op {
            // 可以按范围处理，返回rust的范围表达式
            Ok(range) => Expression::Range(range),
            // 否则，按表达式处理
            Err(_) => {
                let mut tokens = Tokens::new(stream);
                logic(&mut tokens, del)?
            }
        };
        result.push(exp);    
    }
   
    Ok(result)
}

/// 选择字段:= .名字 | .函数调用 | .(名字，名字) | .total(名字，名字)
/// 如果是获得迭代本身的聚集函数内容，加上total标识符，进行识别。
/// 返回：选择内容，选择内容可以是属性，也可以是函数调用，也可以是一批内容
fn select(tokens: &mut Tokens, del: &Vec<String>) -> Result<Select, String> {
    match get_spec_puncts(tokens, &vec!["."]) {
        // 是"."，取选择部分，
        Some(_) => {
            let token = tokens.next().expect(&format!("lib.select: 没取到单词, {}", tokens.to_string()));
            match token {
                TokenTree::Group(ref group) => {
                    match group.delimiter() {
                        // 小括号，返回表达式
                        Delimiter::Parenthesis => {
                            let mut tokens = Tokens::new(group.stream());
                            let selects_exp = array(&mut tokens, |tokens| {
                                attr(tokens, del)
                            })?;
                            Ok(Select::Selects(false, selects_exp))
                        },
                        // 不是小括号，返回空
                        _ => Err(String::from("选择内容只能是小括号"))
                    }
                },
                TokenTree::Ident(ref ident) => {
                    let name_str = ident.to_string();
                    // 如果是total，按选择迭代自己的聚集处理
                    if name_str == "total" {
                        let stream = get_group(tokens, Delimiter::Parenthesis)?;
                        let mut tokens = Tokens::new(stream);
                        let selects_exp = array(&mut tokens, |tokens| {
                            attr(tokens, del)
                        })?;
                        Ok(Select::Selects(true, selects_exp))
                    } else {
                        // 单个内容选择，可能是属性，也可能是函数
                        let name: syn::Ident = syn::parse_str(&name_str).expect("选择内容必须是标识符开始");
                        let select = select_one(tokens, name, del)?;
                        Ok(select)
                    }
                },
                _ => Err(String::from("选择内容只能是标识符或小括号"))
            }
        },
        // 否则，返回无选择部分
        None => {
            Ok(Select::Selects(false, Vec::new()))
        }
    }
}

/// 取到一项选择内容及别名
fn attr(tokens: &mut Tokens, del: &Vec<String>) -> Result<(syn::Ident, Expression), String> {
    // 调用表达式解析，得到表达式内容
    let exp = logic(tokens, del)?;
    // 取得别名
    let token = tokens.next().expect(&format!("必须有别名, {:?}", tokens.to_string()));
    let name = to_ident(&token)?;
    Ok((name, exp))
}

// 单个选择项:= 名称 | 函数名(函数参数)
// @param ident: 属性名，或者函数名
// return：表达式及聚集函数名，如果是聚集函数，函数里的表达式当表达式，函数名本身单独返回，有些聚集函数没有参数，表达式可能为空
fn select_one(tokens: &mut Tokens, name: syn::Ident, del: &Vec<String>) -> Result<Select, String> {
    // 取单词，如果不是"()"，就是取属性
    match get_group(tokens, Delimiter::Parenthesis) {
        Ok(stream) => {
            let (exp, name) = call(&mut Tokens::new(stream), name, del)?;
            // 返回聚集函数调用的选择
            Ok(Select::Agg(name.expect("聚集函数必须有名字"), exp))
        }, 
        // 不是小括号，按属性返回
        Err(_) => {
            Ok(Select::Field(name))
        }
    }
}

// 解决这种文法，表达式 := 数字 ((+|-)数字)*
// @param parser: 执行解析的解析器。
// @param tokens: 单词流。
// @param opers: 需要满足的二元操作符
// @return: AST语法树
pub fn binary<F>(tokens: &mut Tokens, opers: &Vec<&str>, parser: F) -> Result<Expression, String>
where F: Fn(&mut Tokens) -> Result<Expression, String> {
    let mut acc = parser(tokens)?;
    // 让解析器不断工作，直到出错为止
    loop {
        let ident = get_spec_puncts(tokens, opers);
        match ident {
            // 不是给定标识符，结束
            None => break,
            Some(ident) => {
                let right = parser(tokens)?;
                acc = Expression::Binary(ident, Box::new(acc), Box::new(right))
            }
        }
    }

    Ok(acc)
}

// 解决这种文件，函数参数 := (表达式 (,表达式)*)?，分隔符先假定为逗号
pub fn array<F, T>(tokens: &mut Tokens, parser: F) -> Result<Vec<T>, String>
where F: Fn(&mut Tokens) -> Result<T, String> {
    // 如果没有内容，返回空
    if tokens.is_empty() {
        Ok(Vec::new())
    } else {
        let param = parser(tokens)?;
        let mut result = vec![param];
        // 当下一个单词是逗号时，一直做
        while let Some(_) = get_spec_puncts(tokens, &vec![","]) {
            // 取下一个参数
            result.push(parser(tokens)?);
        }
        Ok(result)
    }
}

// 取给定标识符，没取到，返回空，单词回滚
fn get_spec_puncts(tokens: &mut Tokens, strs: &Vec<&str>) -> Option<String> {
    let oper = get_puncts(tokens);
    // 包括在给定标识符内，返回取到的结果，否则，把标识符字符一个个回退
    if strs.contains(&oper.as_str()) {
        Some(oper)
    } else {
        // 非空字符串，才会回退
        if !oper.is_empty() {
            // 把标识符包含的字符一个个回退，前面的，按有后续内容处理
            for c in oper[0..(oper.len()-1)].chars() {
                let punct = TokenTree::Punct(proc_macro::Punct::new(c, proc_macro::Spacing::Joint));
                tokens.tokens.push(punct);
            }
            // 回退标识符最后一个字符，按无后续内容处理
            let punct = TokenTree::Punct(proc_macro::Punct::new(oper.chars().last().expect("这时肯定有值"), proc_macro::Spacing::Alone));
            tokens.tokens.push(punct);
        }
        None
    }
}

// 获取操作符组合结果
fn get_puncts(tokens: &mut Tokens) -> String {
    let mut result = String::new();

    // 获取单个标识符
    match get_punct(tokens) {
        None => {},
        Some(punct) => {
            result.push(punct.as_char());
            // 后面还有标识符
            if let proc_macro::Spacing::Joint = punct.spacing() {
                let ch = get_punct(tokens).unwrap().as_char();
                result.push(ch);
            }
        }
    }
    result
}

// 获取单个操作符，不是操作符，回退单词
fn get_punct(tokens: &mut Tokens) -> Option<Punct> {
    let token_op = tokens.next();
    match token_op {
        None => None,
        Some(token) => {
            match token {
                // 有单词，看是否操作符，不是返回空
                TokenTree::Punct(punct) => Some(punct),
                _ => {
                    tokens.tokens.push(token);
                    None
                }
            }
        }
    }
}

// 单词是否给定标识符
fn is_key(token: &TokenTree, key: &str) -> bool {
    match token {
        TokenTree::Ident(ident) => {
            ident.to_string() == key
        },
        _ => false
    }
}

/// 取标识符，如果不是标识符，回退单词，返回错误
fn get_ident(tokens: &mut Tokens) -> Result<String, String> {
    // 取一个单词
    let token_op = tokens.next();
    let result = match token_op {
        Some(ref token) => {
            match token {
                TokenTree::Ident(ref ident) => Some(ident.to_string()),
                _ => None
            }
        },
        None => None,
    };

    // 如果结果为空，把单词回退
    match result {
        Some(x) => Ok(x),
        None => {
            let err = format!("{:?}", &token_op);
            tokens.back_token(token_op);
            Err(format!("需要标识符，实际是：{}", err))
        }
    }
}

/// 取给定标识符，即使是标识符，但不是给定的，也要回退单词，返回错误
fn get_spec_ident(tokens: &mut Tokens, idents: &Vec<&str>) -> Result<String, String> {
    // 取一个单词
    let token_op = tokens.next();
    let result = match token_op {
        Some(ref token) => {
            match token {
                TokenTree::Ident(ref ident) => {
                    let str = ident.to_string();
                    // 如果标识符在给定范围内，返回有值
                    if idents.contains(&str.as_str()) {
                        Some(str)
                    } else {
                        None
                    }
                },
                _ => None
            }
        },
        None => None,
    };

    // 如果结果为空，把单词回退
    match result {
        Some(x) => Ok(x),
        None => {
            let err = format!("{:?}", &token_op);
            tokens.back_token(token_op);
            Err(format!("需要标识符，实际是：{}", err))
        }
    }
}

// 获取给定分组表示的分组内容，如果不是给定分组，单词回退，返回空。
// 如果单词为空，返回空，不用回退。
// return: 分组内容转换成的单词流，如果没有取到所需单词，错误显示所需单词及实际单词
fn get_group(tokens: &mut Tokens, delimiter: proc_macro::Delimiter) -> Result<TokenStream, String> {
    // 取一个单词
    let token_op = tokens.next();
    let result = match token_op {
        Some(ref token) => {
            match token {
                TokenTree::Group(ref group) => {
                    // 是给定分组符号，返回内容转换成单词流的结果
                    if group.delimiter() == delimiter {
                        Some(group.stream())
                    } else {
                        None
                    }
                },
                _ => {
                    None
                }
            }
        },
        None => None,
    };

    // 如果结果为空，把单词回退
    match result {
        Some(x) => Ok(x),
        None => {
            let err = format!("{:?}", &token_op);
            tokens.back_token(token_op);
            Err(format!("需要：{:?}，实际是：{}", delimiter, err))
        }
    }
}

// 取标识符
fn to_ident(token: &TokenTree) -> Result<syn::Ident, String> {
    match token {
        TokenTree::Ident(ident) => {
            let ident: syn::Ident = syn::parse_str(&ident.to_string()).expect("标识符要合法");
            Ok(ident)
        },
        _ => Err(format!("必须是标识符: {:?}", token.to_string()))
    }
}*/
