fn make_ascii_titlecase(s: &mut str) -> &str {
    if let Some(r) = s.get_mut(0..1) {
        r.make_ascii_uppercase();
    }
    s
}

fn sum_required_fuel(input_string: &str, fuel_calculator: fn(u32) -> u32) -> Result<u32, String> {
    let parts = input_string
        .lines()
        .enumerate()
        .map(|(line_index, line)| {
            let module_mass = line.parse::<u32>().map_err(|error| {
                format!(
                    "Line {}: {}",
                    line_index + 1,
                    make_ascii_titlecase(&mut error.to_string())
                )
            })?;
            if module_mass < 6 {
                return Err(format!(
                    "Line {}: Too small module mass (less than 6)",
                    line_index + 1
                ));
            }
            Ok(fuel_calculator(module_mass))
        })
        .collect::<Result<Vec<u32>, String>>()?;
    Ok(parts.iter().sum())
}

pub fn part1(input_string: &str) -> Result<u32, String> {
    sum_required_fuel(input_string, |mass| mass / 3 - 2)
}

pub fn part2(input_string: &str) -> Result<u32, String> {
    fn required_fuel(mass: u32) -> u32 {
        match (mass / 3).checked_sub(2) {
            Some(fuel) => fuel + required_fuel(fuel),
            None => 0,
        }
    }

    sum_required_fuel(input_string, required_fuel)
}

#[test]
pub fn tests_part1() {
    assert_eq!(Ok(2), part1("12"));
    assert_eq!(Ok(2), part1("14"));
    assert_eq!(Ok(654), part1("1969"));
    assert_eq!(Ok(33583), part1("100756"));

    assert_eq!(Ok(3_262_358), part1(include_str!("day01_input.txt")));

    assert_eq!(
        Some("Line 1: Cannot parse integer from empty string".to_string()),
        part1("\n").err()
    );
}

#[test]
fn tests_part2() {
    assert_eq!(Ok(2), part2("14"));
    assert_eq!(Ok(966), part2("1969"));
    assert_eq!(Ok(50346), part2("100756"));

    assert_eq!(Ok(4_890_696), part2(include_str!("day01_input.txt")));
}
