extern crate proc_macro;

use proc_macro::TokenStream;
use proc_macro2::TokenStream as TokenStream2;

use quote::quote;
use syn::parse_macro_input;

#[proc_macro_attribute]
pub fn access_control(args: TokenStream, item: TokenStream) -> TokenStream {
    let item = parse_macro_input!(item as syn::ItemStruct);

    if args.is_empty() {
        let item_ident = &item.ident;
        let item_vis = &item.vis;
        let fields = &item.fields;

        let struct_name_to_generate = format!("{}AccessControlRoleData", item_ident);

        let storage_name_to_generate = format!("StorageKeyAccessControl{}", item_ident);

        let roles_struct_field_name_generate =
            format!("{}_access_control_roles", item_ident).to_lowercase();

        let struct_ident = syn::Ident::new(&struct_name_to_generate, item_ident.span());

        let storage_key_ident = syn::Ident::new(&storage_name_to_generate, item_ident.span());

        let roles_struct_field_ident =
            syn::Ident::new(&roles_struct_field_name_generate, item_ident.span());

        let mut res = TokenStream2::new();

        let mut field_vec = vec![];

        for field in fields.iter() {
            let mut field_tk = quote! {#field};
            field_tk.extend(quote! {,});

            field_vec.push(field_tk);
        }

        res.extend(field_vec);

        let ts = quote! {

            pub enum #storage_key_ident {
                Roles,
                AdminRole(std::string::String),
                RoleData(std::string::String)
            }

            impl #storage_key_ident {
                pub fn to_string(&self) -> std::string::String {
                    match self {
                        #storage_key_ident::Roles => "rol".to_string(),
                        #storage_key_ident::AdminRole(adm) => format!("{}adm", adm),
                        #storage_key_ident::RoleData(data) => format!("{}data", data),
                    }
                }

                pub fn into_bytes(&self) -> std::vec::Vec<u8> {
                    self.to_string().into_bytes()
                }
            }

            #[derive(near_sdk::borsh::BorshDeserialize, near_sdk::borsh::BorshSerialize)]
            pub struct #struct_ident {
                members: near_sdk::collections::LookupSet<near_sdk::AccountId>,
                admin_role: near_sdk::collections::LookupMap<std::string::String, std::string::String>,
         }

             #[derive(near_sdk::borsh::BorshDeserialize, near_sdk::borsh::BorshSerialize, near_sdk::PanicOnDefault)]
             #item_vis struct #item_ident {
                #roles_struct_field_ident: near_sdk::collections::UnorderedMap<std::string::String, #struct_ident>,
                    #res
                }


                macro_rules! init_roles {(
                    Self{
                        $($field_name:ident: $field_type:expr,)*
                    }
                ) => {

                    Self {
                        $($field_name: $field_type,)*
                        #roles_struct_field_ident: near_sdk::collections::UnorderedMap::new(#storage_key_ident::Roles.into_bytes())
                    }
                }
            }


                #[near_bindgen]
                impl AccessControl for #item_ident {


                    fn add_role(&mut self, role: &std::string::String) {

                        self.assert_self();
                        // Check that role is not already registered
                        if self.#roles_struct_field_ident.get(role).is_none() {
                            let mut role_data = #struct_ident {
                                members: near_sdk::collections::LookupSet::new(#storage_key_ident::RoleData(role.to_string()).into_bytes()),
                                admin_role: near_sdk::collections::LookupMap::new(#storage_key_ident::AdminRole(role.to_string()).into_bytes()),
                            };

                            role_data
                                .admin_role
                                .insert(role, &"default_admin".to_string());
                            self.#roles_struct_field_ident.insert(role, &role_data);

                            near_sdk::env::log(format!("Role {} is added", role).as_bytes())
                        }
                    }

                    fn has_role(&self, role: &std::string::String, account: &near_sdk::AccountId) -> bool {
                        let role_data = self.#roles_struct_field_ident.get(role);

                        match role_data {
                            Some(r) => r.members.contains(account),
                            None => near_sdk::env::panic(format!("Role: {} does not exist", role).as_bytes())
                        }
                    }


                    fn check_role(&self, role: &String, account: &AccountId) {
                        if !self.has_role(role, account) {
                            env::panic(format!("Account {} , is missing: {} role", account, role).as_bytes());
                        }
                    }


                    fn assert_role(&self, role: &String) {
                        self.check_role(role, &near_sdk::env::predecessor_account_id())
                    }

                    fn assert_self(&mut self) {
                        if env::predecessor_account_id() != env::current_account_id() {
                            env::panic("Function is private".as_bytes())
                        }
                    }

                    fn get_role_admin(&self, role: &std::string::String) -> std::string::String {
                        let role_data = self.#roles_struct_field_ident.get(role);

                        match role_data {
                            Some(r) => {

                                r.admin_role.get(role).unwrap().to_string()
                            }
                            None => near_sdk::env::panic(format!("Role: {} does not exist", role).as_bytes())
                        }
                    }

                    fn get_account_roles(&self, account: &near_sdk::AccountId) -> std::vec::Vec<std::string::String> {
                        let mut found_role = std::vec::Vec::new();

                        for role in self.#roles_struct_field_ident.keys() {
                            if self.has_role(&role, account) {
                                found_role.push(role);
                            }
                        }

                        found_role
                    }

                    fn grant_role(&mut self, role: &String, account: &AccountId) {
                        self.assert_role(&self.get_role_admin(role));
                        self.add_role_member(role, account);
                    }


                    fn setup_account_role(&mut self, role: &String, account: &AccountId) {
                        self.assert_self();

                        self.add_role(role);
                        self.add_role_member(role, account);
                    }

                    fn delete_role_member(&mut self, role: &std::string::String, account: &near_sdk::AccountId) {
                        self.assert_self();

                        if self.has_role(role, account) {
                            let role_data = self.#roles_struct_field_ident.get(role);

                            match role_data {
                                Some(mut r) => {
                                    r.members.remove(account);

                                    near_sdk::env::log(format!("Role {} is revoked from {}", role, account).as_bytes())

                                }
                                None => near_sdk::env::panic(format!("Role: {} does not exist", role).as_bytes()),
                            }
                        }
                    }

                    fn revoke_role(&mut self, role: &String, account: &AccountId) {
                        self.assert_role(&self.get_role_admin(role));

                        self.delete_role_member(role, account)
                    }

                    fn set_admin_role(&mut self, role: &std::string::String, admin_role: &std::string::String) {

                        self.assert_role(&self.get_role_admin(role));

                        if self.get_role_admin(role) != *admin_role {
                            let role_data = self.#roles_struct_field_ident.get(role);

                            match role_data {
                                Some(mut r) => {
                                    r.admin_role.get(role).unwrap().clear();

                                    r.admin_role.insert(role, &admin_role.to_string());

                                    near_sdk::env::log(
                                        format!(
                                            "Changed admin role from: {}. To: {}",
                                            r.admin_role.get(role).unwrap(),
                                            admin_role
                                        )
                                        .as_bytes(),
                                    );
                                }
                                None => near_sdk::env::panic(format!("Role: {} does not exist", role).as_bytes()),
                            }
                        }
                    }

                    fn add_role_member(&mut self, role: &std::string::String, account: &near_sdk::AccountId) {

                        self.assert_self();
                        if !self.has_role(role, account) {
                            let role_data = self.#roles_struct_field_ident.get(role);

                            near_sdk::env::log(format!("Setting role: {}. To: {}", role, account).as_bytes());

                            match role_data {
                                Some(mut r) => {
                                    r.members.insert(account);

                                    near_sdk::env::log(format!("Account {} is added to {}", account, role).as_bytes())

                                }
                                None => near_sdk::env::panic(format!("Role: {} does not exist", role).as_bytes()),
                            }
                        }
                    }

                    }


        };
        ts.into()
    } else {
        quote! {compile_error!("access_control does not take any arguments");}.into()
    }
}
