use near_sdk::{env, AccountId};

pub trait AccessControl {
    //HELPER - SHOULD REMAIN PRIVATE
    fn add_role(&mut self, role: &String);

    // CAN BE MADE ACCESSIBLE
    fn has_role(&self, role: &String, account: &AccountId) -> bool;

    // CAN BE MADE ACCESSIBLE
    fn check_role(&self, role: &String, account: &AccountId) {
        if !self.has_role(role, account) {
            env::panic(format!("Account {} , is missing: {} role", account, role).as_bytes());
        }
    }

    // CAN BE MADE ACCESSIBLE
    fn get_role_admin(&self, role: &String) -> String;

    // CAN BE MADE ACCESSIBLE
    fn get_account_roles(&self, account: &AccountId) -> Vec<String>;

    // CAN BE MADE ACCESSIBLE
    fn grant_role(&mut self, role: &String, account: &AccountId) {
        self.assert_role(&self.get_role_admin(role));
        self.add_role_member(role, account);
    }

    // SHOULD REMAIN PRIVATE
    fn setup_account_role(&mut self, role: &String, account: &AccountId) {
        self.add_role(role);
        self.add_role_member(role, account);
    }

    //HELPER - SHOULD REMAIN PRIVATE
    fn delete_role_member(&mut self, role: &String, account: &AccountId);

    // CAN BE MADE ACCESSIBLE
    fn revoke_role(&mut self, role: &String, account: &AccountId) {
        self.assert_role(&self.get_role_admin(role));

        self.delete_role_member(role, account)
    }

    // CAN BE MADE ACCESSIBLE
    fn set_admin_role(&mut self, role: &String, admin_role: &String);

    // CAN BE MADE ACCESSIBLE
    fn assert_role(&self, role: &String) {
        self.check_role(role, &env::predecessor_account_id())
    }

    //HELPER - SHOULD REMAIN PRIVATE
    fn add_role_member(&mut self, role: &String, account: &AccountId);
}
