# Access Control NEAR

Access Control NEAR is a library for implementing role-based access control model in NEAR smart contracts. It is partially inspired by OpenZeppelin's access control implementation. 

# Architecture

The **Roles** map consists of **Role** to **Role Data** mapping. New members are added to the **members** set by inserting new **AccountId**. Each role has an **Admin Role**, whose members are allowed to perform privileged actions on the role that derives it. Default **admin_role** for all created roles is `default_admin` 

![diagram1.png](images/diagram1.png)

![diagram2.png](images/diagram2.png)

## Methods

All methods, by default, are private. The reason behind it is to let the user allow methods that are necessary to be used on-chain. Some methods should be kept private since they carry privileged actions that anyone can call. Therefore, they should be only used within the smart contract if necessary. 

You can find the list of all available methods below. **Private** or **Helper** should **NOT** be made public to be used on-chain. **Public** methods can be made public, respectively. 

### Public Methods

```rust
fn has_role(&self, role: &String, account: &AccountId) -> bool;

fn check_role(&self, role: &String, account: &AccountId);

fn get_role_admin(&self, role: &String) -> String;

fn get_account_roles(&self, account: &AccountId) -> Vec<String>;

fn grant_role(&mut self, role: &String, account: &AccountId);

fn revoke_role(&mut self, role: &String, account: &AccountId);

fn set_admin_role(&mut self, role: &String, admin_role: &String);

fn assert_role(&self, role: &String);
```

- **has_role**: Checks if given account has the given role. Returns bool
- **check_role**: Checks if given account has the given role. Panics with a message
- **get_role_admin**: View method. Gets the admin role of a given role. Returns String
- **get_account_roles**: View method. Gets all roles that the given account has. Returns a vector containing all the roles
- **grant_role**: Can only be called by the role admin of given role. Grants given role to given account.
- **revoke_role**: Can only be called by the role admin of given role. Revokes given role for given account.
- **set_admin_role**:  Can only be called by the role admin of given role. Sets the new admin role for given role.
- **assert_role**: Checks whether the caller has given role. Panics with a message. Internally calls **check_role** with `env::predecessor_account_id()`

### Private Methods

```rust
fn setup_account_role(&mut self, role: &String, account: &AccountId);
```

- **setup_account_role**: Sets the given role to given account. Can be called by anyone. Therefore, should remain private. If role does not exist it first creates by calling **add_role.**

### Helper Methods

```rust
fn add_role(&mut self, role: &String);

fn delete_role_member(&mut self, role: &String, account: &AccountId);

fn add_role_member(&mut self, role: &String, account: &AccountId);
```

- **add_role**: Adds a new role and sets `default_admin` as the **admin_role**
- **delete_role_member**: Deletes a member from a role.
- **add_role_member**: Adds a new member to a role.

# Usage

You can run the test application in the **test** folder, which is a fork of **StatusMessage** by calling `./build.sh` and then `./deploy.sh` . Please update `./deploy.sh` to have your accounts.

The only thing needed is to add `#[access_control]` attribute macro to your main struct to begin using methods from this library. Please also note that `#[access_control]`  macro already includes `#[derive(BorshDeserialize, BorshSerialize, PanicOnDefault)]` . Therefore, please do not derive it the second time on your main struct, where the `#[access_control]` is used. 

```rust
#[near_bindgen]
#[access_control]
pub struct StatusMessage {
    records: LookupMap<String, String>,
}
```

Then, to begin using methods in the Access Control NEAR and setup initial roles, you have to first call the `init!()` macro in your constructor and then setup roles you want to use. 

```rust
const DEFAULT_ADMIN: &str = "default_admin";
const MINTER: &str = "minter";
const MANAGER: &str = "manager";

.....
#[near_bindgen]
impl StatusMessage {
    #[init]
    pub fn new(owner: AccountId, minter: AccountId, manager: AccountId) -> Self {
        assert!(!env::state_exists(), "The contract is already initialized.");

        let mut constructor = init!(Self {
            records: LookupMap::new(StorageKey::Records.into_bytes()),
        });

        constructor.setup_account_role(&DEFAULT_ADMIN.to_string(), &owner);
        constructor.setup_account_role(&MINTER.to_string(), &minter);
        constructor.setup_account_role(&MANAGER.to_string(), &manager);

        constructor.set_admin_role(&MINTER.to_string(), &MANAGER.to_string());

        constructor
    }
```

That's all! From now on, you can directly use AC NEAR methods within your smart contract. To make some methods public, you can add a public wrapper around them. 

```rust
pub fn pub_get_account_roles(&self, account: &AccountId) -> Vec<String> {
        self.get_account_roles(account)
    }

    pub fn pub_grant_role(&mut self, role: &String, account: &AccountId) {
        self.grant_role(role, account)
    }

    pub fn pub_has_role(&self, role: &String, account: &AccountId) -> bool {
        self.has_role(role, account)
    }

    pub fn pub_check_role(&self, role: &String, account: &AccountId) {
        self.check_role(role, account)
    }

    pub fn pub_get_role_admin(&self, role: &String) -> String {
        self.get_role_admin(role)
    }

    pub fn pub_revoke_role(&mut self, role: &String, account: &AccountId) {
        self.revoke_role(role, account)
    }

    pub fn pub_set_admin_role(&mut self, role: &String, admin_role: &String) {
        self.set_admin_role(role, admin_role)
    }

    pub fn pub_assert_role(&self, role: &String) {
        self.assert_role(role)
    }
```

# Future Plans

- Finishing up tests
- Doing an audit for this library
- Making stricter version, where one account is only allowed to have one role

# Licence

- [MIT](https://choosealicense.com/licenses/mit/)