use super::{bilinear_map::bls12_381::Bls12_381, gpsw::abe::Gpsw, AccessPolicy, Engine, Policy};
use crate::{error::FormatErr, policy::ap};

/// # Encryption using an Authorization Policy
/// This test demonstrates how data can be encrypted with policy attributes.
/// An user will only be able to decrypt data when it holds a key with the
/// proper attributes. This test also demonstrates revocation of an
/// attribute value and how to implement forward secrecy.
#[test]
fn abe() -> Result<(), FormatErr> {
    // ## Policy
    // In this demo, we will create a Policy which combines two axes, a
    // 'security level' and a 'department'. An user will be able to decrypt
    // data only if it possesses a key with a sufficient security level
    // or the attribute for the department.
    // The parameter fixes the maximum number of revocations of attributes (see
    // below) for this Policy. This number influences the number of
    // public keys which will be ultimately generated for this Security Group
    // and must be kept to a "reasonable" level to reduce security risks associated
    // with multiplying the number of keys.

    // ## Policy Axes
    // The Policy is defined by two Policy Axes, thus defining a 2 dimensional
    // matrix of authorizations. An user must possess keys with attributes
    // from these two axes to be able to decrypt files. ### Security Level
    // Axis The first Policy Axis is the 'Security Level' axis and is a
    // hierarchical axis made of 5 levels: level 1, ̀level 2̀ , ..., level 5̀
    // It is hierarchical: a user being granted access to level `n` is automatically
    // granted access to all levels below n.
    // ### Department Security Axis
    // The second Policy Axis is the Department axis and is made of 4 values: R&D,
    // HR, MKG, FIN. This axis is not hierarchical: granting access to an
    // attribute of this axis to a user does not give access to any other
    // attribute. Each attribute must be granted individually.
    let policy = Policy::new(1000)
        .add_axis(
            "Security Level",
            &["level 1", "level 2", "level 3", "level 4", "level 5"],
            true,
        )?
        .add_axis("Department", &["R&D", "HR", "MKG", "FIN"], false)?;

    // ## Master Authority
    // The Master Authority posesses the keys for the given Policy:
    // a Secret Key which is used to delegate authority to "delegate authorities",
    // which are the ones generating user keys and a Public key which is
    // used to encrypt files with proper level of security.
    let mut engine = Engine::<Gpsw<Bls12_381>>::new(&policy);
    let (priv_key, pub_key, del_key) = engine.generate_master_key()?;

    // ## Delegate Authorities
    // The Master Authority will delegate part or all of its authority to "Delegate
    // Authorities" (a.k.a Delegates) which are the ones generating
    // user keys.
    // In this particular example, the Master Authority will delegate its authority
    // to 2 Delegates:
    //  - a Super Delegate which can issue User Keys for all Security Levels and all
    //    Departments
    //  - a "Level 4 Marketing and Finance Delegate" which can only generate User
    //    Keys for marketing (MKG) or finance (FIN) data of Security Level 4 and
    //    below
    let super_delegate = engine.generate_user_key(&priv_key, &AccessPolicy::All)?;
    let access_policy =
        ap("Security Level", "level 4") & (ap("Department", "MKG") | ap("Department", "FIN"));
    let level_4_mkg_fin_delegate = engine.generate_user_key(&priv_key, &access_policy)?;

    // ## User Keys
    // Delegate Authorities can now generate User Keys up to the level allowed by
    // their policy. A marketing user with level 3 security can have its key
    // generated by any of the Delegates.
    let access_policy = ap("Security Level", "level 3") & ap("Department", "MKG");
    let _level_3_mkg_user = engine.delegate_user_key(&del_key, &super_delegate, &access_policy)?;
    let level_3_mkg_user =
        engine.delegate_user_key(&del_key, &level_4_mkg_fin_delegate, &access_policy)?;

    // However, a Delegate cannot generate user keys for which it does not have the
    // authority
    let access_policy = ap("Security Level", "level 5") & ap("Department", "MKG");
    let level_5_user =
        engine.delegate_user_key(&del_key, &level_4_mkg_fin_delegate, &access_policy);
    // FAILURE: as expected the level 4 marketing authority cannot generate user
    // keys for Security Level 5
    assert!(level_5_user.is_err());

    let access_policy = ap("Security Level", "level 3") & ap("Department", "HR");
    let hr_user = engine.delegate_user_key(&del_key, &level_4_mkg_fin_delegate, &access_policy);
    // FAILURE: as expected the level 4 marketing authority cannot generate user
    // keys for Department HR
    assert!(hr_user.is_err());

    // Let us create a super user as well, which can decrypt everything
    // Note that super_user have just a randomization of super_delegate key but the
    // private_key is not needed anymore
    let super_user = engine.delegate_user_key(&del_key, &super_delegate, &AccessPolicy::All)?;

    // ## Encryption and Decryption
    // Data is encrypted using the Master Authority Public Key with two attributes:
    // one for the Security Level and one for the Department.
    // Anyone - who has access to the Public Key - can encrypt data with any
    // attribute combination. However, only users posessing user keys with
    // the right combination of attributes can decrypt data.

    // ### A level 2 marketing message
    // Let us create an encrypted marketing message with a security level 2.
    let (plain_text_keep, cipher_text_keep) = engine.generate_cleartext_ciphertext(
        &[
            ("Security Level", "level 2").into(),
            ("Department", "MKG").into(),
        ],
        &pub_key,
    )?;

    // Both users are able to decrypt the message
    let result = engine
        .decrypt(&cipher_text_keep, &level_3_mkg_user)?
        .expect("Decryption must works");
    assert_eq!(plain_text_keep, result, "level_3_mkg_user");
    let result = engine
        .decrypt(&cipher_text_keep, &super_user)?
        .expect("Decryption must works");
    assert_eq!(plain_text_keep, result, "super_user_lvl2_mkg");

    // ### A level 5 marketing message
    // However in the case of a level 5 marketing message, only the super user will
    // succeed:
    let (pt, ct) = engine.generate_cleartext_ciphertext(
        &[
            ("Security Level", "level 5").into(),
            ("Department", "MKG").into(),
        ],
        &pub_key,
    )?;
    let result = engine.decrypt(&ct, &level_3_mkg_user)?;
    assert!(result.is_none(), "level_3_mkg_user");
    let result = engine
        .decrypt(&ct, &super_user)?
        .expect("Decryption must works");
    assert_eq!(pt, result, "super_user_lvl5");

    // ### A level 2 HR message
    // Likewise, in the case of a level 2 HR message, only the super user will
    // succeed:
    let (pt, ct) = engine.generate_cleartext_ciphertext(
        &[
            ("Security Level", "level 2").into(),
            ("Department", "HR").into(),
        ],
        &pub_key,
    )?;
    let result = engine.decrypt(&ct, &level_3_mkg_user)?;
    assert!(result.is_none(), "level_3_mkg_user");
    let result = engine
        .decrypt(&ct, &super_user)?
        .expect("Decryption must works");
    assert_eq!(pt, result, "super_user_lvl2_hr");

    // ## Revocation
    // At anytime the Master Authority can revoke an attribute.
    // When that happens future encryption of data for a given attribute cannot be
    // decrypted with keys which are not "refreshed" for that attribute. Let
    // us revoke the Security Level 2
    engine.update(&("Security Level", "level 2").into())?;
    // We now encrypt a new marketing message at level 2
    let (pt, ct) = engine.generate_cleartext_ciphertext(
        &[
            ("Security Level", "level 2").into(),
            ("Department", "MKG").into(),
        ],
        &pub_key,
    )?;
    // No user, can decrypt the message until its key is refreshed
    let result = engine.decrypt(&ct, &level_3_mkg_user)?;
    assert!(result.is_none(), "level_3_mkg_user");
    let result = engine
        .decrypt(&ct, &super_user)?
        .expect("Decryption must works");
    // works as MKG is present
    assert_eq!(pt, result, "super_user_lvl2_rvk");
    // All keys need to be refresh: Delegates and Users
    // Delegates
    let super_delegate = engine.generate_user_key(&priv_key, &AccessPolicy::All)?;
    let access_policy: AccessPolicy =
        ap("Security Level", "level 4") & (ap("Department", "MKG") | ap("Department", "FIN"));
    let level_4_mkg_fin_delegate = engine.generate_user_key(&priv_key, &access_policy)?;
    // Users
    let access_policy: AccessPolicy = ap("Security Level", "level 3") & ap("Department", "MKG");
    let level_3_mkg_user =
        engine.delegate_user_key(&del_key, &level_4_mkg_fin_delegate, &access_policy)?;
    let super_user = engine.delegate_user_key(&del_key, &super_delegate, &AccessPolicy::All)?;
    // New messages can now be decrypted
    let result = engine
        .decrypt(&ct, &level_3_mkg_user)?
        .expect("Decryption must works");
    assert_eq!(pt, result, "level_3_mkg_user");
    let result = engine
        .decrypt(&ct, &super_user)?
        .expect("Decryption must works");
    assert_eq!(pt, result, "super_user");
    // Older messages can still be decrypted as well
    let result = engine
        .decrypt(&cipher_text_keep, &level_3_mkg_user)?
        .expect("Decryption must works");
    assert_eq!(plain_text_keep, result, "new_level_3_mkg_user");
    let result = engine
        .decrypt(&cipher_text_keep, &super_user)?
        .expect("Decryption must works");
    assert_eq!(plain_text_keep, result, "new_super_user_lvl2_mkg");
    Ok(())
}
