# Error Handling
Core feature of Rust, safe language = safe errors!
 - Recoverable: Expected issues during execution
 - Unrecoverable: Forming from bugs, inadequate accesses,...
Rust does not have EXCEPTIONS (thank god) -> `Result<T, E>` + `panic!`

# `panic!` - Unrecoverable Errors
Panic happens automatically on an unrecoverable error, the prints the failure message, cleans stack and quits.

On the Cargo.toml file you can change the behaviour of `panic`
```rust
[profile.release]
panic = 'abort' // This avoids unwinding error and saves memory but still stops operation
```

You can directly call `panic!("Error message")`

To get fullp backtrace, you have to set `RUST_BACKTRACE=1` as an Environment Variable
https://stackoverflow.com/questions/54055139/how-to-pass-rust-backtrace-1-when-running-a-rust-binary-installed-in-debian

The backtrace shows all the call stack, including functions from Rust stl, crates and similar,...

To get this info it is required you use debug info (debug build or specifying?)
```rust
// In cargo.toml
[profile.release]
debug = true // Adds debug symbols to the profile
```

===

# Result<T,E> - Recoverable Errors

`Result` type is like an `Option`, it has two variants which are 2 different generic parameters, depending on Success or Error:
```rust
enum Result<T, E> {
    Ok(T),
    Err(E),
}

// Ex:
use std::fs::File;

let f = File::open("hello.txt");
let f2: u32 = File::open("hello.txt"); // This will throw an error, as the return is a `Result<T,E>` and has to handle both types

// Result for File::open will be either Ok(file_handle) or Err(more_info) 

let f3 = match File::open("hello.txt") {
    Ok(file) => file,
    Err(error) => panic!("Problem opening file: {:?}", error)
}
```

You can also continue and perform different actions for each error type
```rust
let f3  = match File::open("hello.txt") {
    Ok(file) => file,
    Err(error) => match error.kind() {
        ErrorKind::NotFound => match File::create("hello.txt") {
            Of(fc) => fc,
            Err(e) => panic!("Problem Creating file: {:?}", other_error)
        },
        other_error => panic!("Problem opening the file: {:?}", other_error)
    }
};
```
But there are other ways. The `Result` type has the function `unwrap_or_else` which acts only on errors, if not returns the value in the `Ok` part:
```rust
let f = File::open("hello.txt").unwrap_or_else(|error| { // |error| is declaring how to call the error variable, then we start a scope
        if error.kind() == ErrorKind::NotFound {
            File::create("hello.txt").unwrap_or_else(|error| { // same here
                panic!("Problem creating the file: {:?}", error);
            })
        } else {
            panic!("Problem opening the file: {:?}", error);
        }
    });
// Further in Chapter 13
```

## Shortcuts for Panic - `unwrap` and `expect`
`match` is good, but verbose and might be too general to communicate intent.
All types have different helpers, `Result` does too.
As explained, `unwrap` returns directly the value inside the `Ok` part if it is a success; but panics directly on `Err`.
`unwrap_or_else` executes the scope you put on `Err`.
```rust
let f = File::open("hello.txt").unwrap();
// becomes
thread 'main' panicked at 'called `Result::unwrap()` on an `Err` value: Error {
repr: Os { code: 2, message: "No such file or directory" } }',
src/libcore/result.rs:906:4
```


`expect` is similar but you can set a first error message, then having the one coming from `Err`.
```rust
let f = File::open("hello.txt").expect("Failed to open hello.txt");
// becomes
thread 'main' panicked at 'Failed to open hello.txt: Error { repr: Os { code:
2, message: "No such file or directory" } }', src/libcore/result.rs:906:4
```
This makes it easier to trace where the `panic!` is coming from.

# Propagating Errors
Instead of handling, you can also return the error, then handle it somewhere else
```rust
let mut f = match File::open("hello.txt") {
    Ok(file) => file, // we set file into f for later use
    Err(e) => return Err(e), // Closes the scope and returns an error
};

let mut s = String::new();

match f.read_to_string(&mut s) {
    Ok(_) => Ok(s), // ""
    Err(e) => Err(e), // if we are at the end, this is correct, else `return Err(e)` do achieve the same as previously
}
```

## Shorthand error progragation - `?`
```rust
use std::fs::File;
use std::io;
use std::io::Read;

fn read_username_from_file() -> Result<String, io::Error> {
    let mut f = File::open("hello.txt")?; // // If error -> `return Err(error)`
    let mut s = String::new();
    f.read_to_string(&mut s)?;  // "", else continues execution as expected!
    Ok(s)
}
```
The quirk about `?` is that it will try to convert the Generic `Err` into the type you have setup in the `Result<T,E>` error type, usign the `::from` method of the type. So it must be a type that has the `From` trait.
```rust
// We can concatenate ? expressios as the return would still be valid on success!
let mut s = String::new();
File::open("hello.txt")?.read_to_string(&mut s)?;
Ok(s)
```
Even shorter as the `read_to_string` can directly perform the open if you give it a filename instead of a `file_handle`
```rust
use std::fs;
use std::io;
fn read_username_from_file() -> Result<String, io::Error> {
    fs::read_to_string("hello.txt") // Direct open and read file to string, returns whatever happens!
}
```
Just remember that `?` returns something, so it has to match the function return value or must be used in some way or form.

`?` Can also be called on `Option` types. Can it be called for any `Enum`type return?
 - TODO

===

# Panic or not Panic ?
In principle, don't throw Panic (like excepetions). It is rather meant for prototyping and not knowing how to handle things.
In circumstances where a single fail does mean a full failure (tests).

When you know 100% that will will never get an Err.

- If the code will end up in a bad state -> Broken assumption, contradiction or invalid values
- Code relies on being on a good state
- Can't encode info on the types (Chapter 17)

Panic when it is critical for performance to have proper values. Having to do a lot of checks is costly, so you can bypass the checks and throw an error instead of having to handle it which will end up being more costly.