# Patterns and Matching
Another syntax for matching structs of types. Combines :
 - Literals
 - Destructured Arrays, Enums, Structs or Tuples
 - Variables
 - Wildcards
 - Placeholders

Match against a described data type. Compare a pattern to a value, if it matches then we use the value, for example `if let Some() = ...`. When it does not match, code will not run.

# Where can they be used?

We have already been using them!

## `match` Arms
```rust
// They take this form
match VALUE {
    PATTERN => EXPRESSION,
    PATTERN => EXPRESSION,
    PATTERN => EXPRESSION,
}
// Ex: Chapter 6 with the Coin Enum
fn value_in_cents(coin: Coin) -> u8 {
    match coin {
        Coin::Penny => 1,
        Coin::Nickel => 5,
        Coin::Dime => 10,
        Coin::Quarter => 25,
    }
}
```
`match` must take into account ALL types, using `_` to mean any type, but it can't then use the value it matches.

## Conditional `if let`
Basically it uses a `Enum(shadow_var) = Value` as the conditional.
The condition checks if there is a returned type. Then it can continue the `if else` chain if needed!
```rust
let favorite_color: Option<&str> = None;
let is_tuesday = false;
let age: Result<u8, _> = "34".parse();

if let Some(color) = favorite_color {
    println!("Using your favorite color, {}, as the background", color);
} else if is_tuesday {
    println!("Tuesday is green day!");
} else if let Ok(age) = age {
    if age > 30 {
        println!("Using purple as the background color");
    } else {
        println!("Using orange as the background color");
    }
} else {
    println!("Using blue as the background color");
}
```
A `shadow_var` is like a temporal variable, that is local to the branch where condition is completed.
Shadowed variables are only valid inside that context, they can't be used in the same condition like `if let Some(var) = value && var > 10`.

The good and bad, is that they are not exhaustive by default, you check one type at a time of the enum if you want to.

## `while let` Conditional Loop
Same principle as `if let`, we use the `let Enum(var) = value` as the condition:
```rust
let mut stack = Vec::new();
stack.push(1);
stack.push(2);
stack.push(3);

while let Some(top) = stack.pop() { // We pop last value
    println!("{}", top); // Use the popped value
    // At the 3rd pop, we will get a None, so it will not keep going
}
```

## `for` Loops
`for (shadow_var, sv2, ...) in <collection> {...}`
Basically, we obtain the shadown_variables parsing through a `collection` of something.

```rust
let v = vec!['a', 'b', 'c'];

for (index, value) in v.iter().enumerate() { 
    // Convert the iterator into a tuple
    println!("{} is at index {}", value, index);
}
```

## `let` Statements
`let var = value` is atually a pattern.
```rust
let x = 5;
let (x,y,z) = (1,2,3); 
// Compares the right pattern to left pattern and assigns
let w: u32 = 23; 
// Compares pattern to left and checks that right pattern matches

let (u,v) = (4,5,6); // This does not work, patterns don't match
```

## Function Parameters
```rust
fn foo(x: i32) {...}

fn main() {
    let y = 5;
    foo(5); 
    // This is checking the argument to check the parameter pattern!
}
```
