pub mod BlogRust;

pub mod Blog {    
    // Post holds a Trait Obejct that implements a State
    // Held in an Option<T> because 
    pub struct Post {
        state: Option<Box<dyn State>>, // Holds something that can have a State, private so we control what can be created here!
        content: String, // The Official post that is/will be on the blog, private so that the contents can only be changed through correct means
        approvals: u32,
    }

    impl Post {
        pub fn new() -> Post { // Generate a new Draft the intended way!
            Post {
                state: Some(Box::new(Draft {})),
                content: String::new(),
                approvals: 0,
            }
        }

        pub fn add_text(&mut self, text: &str) {
            //let mut edit_text: &String = self.state.as_ref().unwrap().add_text(self);
            //println!("{:?}", self.state.as_ref().unwrap().add_text(self).take());
            if let Some(s) = self.state.as_ref().unwrap().add_text(& mut self.content).take() {
                println!("Tried adding text");
                s.push_str(text);
            }
        }

        pub fn content(&self) -> &str {
            self.state.as_ref().unwrap().content(self)
            // This will call the content method of `State`
            // If there are `Virutal` sub implementation, that will be called
        }

        // Public function to request review of a `Post`
        pub fn request_review(&mut self) {
            // take() obtains the value from Some which State is
            if let Some(s) = self.state.take() { 
                self.state = Some(s.request_review())
            }
        }

        // Public implementation for approve of a `Post`
        pub fn approve(&mut self) {
            if let Some(s) = self.state.take() {
                self.approvals += 1;
                if(self.approvals > 1){
                    self.state = Some(s.approve())
                }
                else {
                    self.state = Some(s)
                }
            }
        }
    }

    trait State {
        // Private function to change a State
        // Only valid for Box<Self> types, and takes ownership as it is intended for the function to change the state!
        fn request_review(self: Box<Self>) -> Box<dyn State>;
        
        // Delegated to be reimplemented, PendingReview will change to Published!
        fn approve(self: Box<Self>) -> Box<dyn State>;

        // Only Publish will reimplement this method
        // Other states will not return the post contents
        fn content<'a>(&self, post: &'a Post) -> &'a str {
            ""
        }

        fn add_text<'a>(&self, post: &'a mut String) -> Option<&'a mut String> {
            None
        }

        fn reject(self: Box<Self>) -> Box<dyn State>;
    }

    // Empty struct to reference a type that is just implementation based
    struct Draft {}

    impl State for Draft {
        // Private implementation that does the state change
        fn request_review(self: Box<Self>) -> Box<dyn State> {
            Box::new(PendingReview {})
        }

        fn approve(self: Box<Self>) -> Box<dyn State> {
            self
        }

        fn reject(self: Box<Self>) -> Box<dyn State> {
            self
        }

        fn add_text<'a>(&self, post: &'a mut String) -> Option<&'a mut String> {
            Some(post)
        }
    }

    // Empty ""
    struct PendingReview {}

    impl State for PendingReview {
        fn request_review(self: Box<Self>) -> Box<dyn State> {
            self
        }

        fn approve(self: Box<Self>) -> Box<dyn State> {
            Box::new(Published {})
        }

        fn reject(self: Box<Self>) -> Box<dyn State> {
            Box::new(Draft {})
        }
    }

    // Empty ""
    struct Published {}

    impl State for Published {
        fn request_review(self: Box<Self>) -> Box<dyn State> {
            self
        }

        fn approve(self: Box<Self>) -> Box<dyn State> {
            self
        }

        // This sub implementation will send the string only when the post is ready to be published!
        fn content<'a>(&self, post: &'a Post) -> &'a str {
            &post.content
        }

        fn reject(self: Box<Self>) -> Box<dyn State> {
            self
        }
    }
}