# Improving I/O Project
Old
```rust
pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    let mut ret: Vec<&str> = Vec::new();
    for line in contents.lines() { //lines() converts into a collection of lines
        if line.contains(query){
            ret.push(line.trim());
        }
    }

    ret
}

impl Config {
    // Function receives a borrowed array/vector of String
    // Returns a Result, with Ok(Config) that holds intent, or error message
    pub fn new(args: &[String]) -> Result<Config, &str> {
        if args.len() < 3{
            return Err("Less than 2 arguments passed")
        }
        let query = args[1].clone();
        let filename = args[2].clone();
        println!("{:?}", env::var("CASE_INSENSITIVE"));
        let case_sensitive = env::var("CASE_INSENSITIVE").is_err();
        Ok(Config { query, filename, case_sensitive})
    }
}
//===============================
// main.rs
use std::env; // Not needed as std can be used directly, but would be shorter
use std::process;
use mini_grep::Config;

fn main() {
    println!("Start the program!");

    // Obtaining the arguments
    let arg_vec: Vec<String> = env::args().collect(); // Convert into a Vector of strings
    let _config = Config::new(&arg_vec).unwrap_or_else(|err| {
        println!("Error parsing arguments: {}", err);
        process::exit(1); // Stop execution with error number
    });

    println!("{:?}", _config);

    mini_grep::run(_config).unwrap_or_else(|err| {
        println!("App Error: {}", err);
        process::exit(1);
    });
}
```
### Step 1 - Use the Iterator from env::args
```rust
pub fn new(mut args: env::Args) -> Result<Config, &str>
// then main
//...
// let arg_vec: Vec<String> = env::args().collect(); // Delete this
    
let _config = Config::new(env::args()).unwrap_or_else(//...
```

### Step 2 - Iterate over values instead of indexing in
```rust
args.next(); // Skip 1st argument
let query = match args.next() {
    Some(arg) => arg,
    None => return Err("Missing arg1"),
}
let filename = match args.next() {
    Some(arg) => arg,
    None => return Err("Missing arg2"),
}
```
Use `match` on the `Option` returned from `next`

### Step 3 - Specify lifetime of return
`'static` why?
Because now we are for sure only return a static string.
Previously lifetime was inferred from the borrowing, both would have the same lifetime and was inferred. Now Args is taken ownership of which makes it not be able to infer lifetime from a borrowed value.
```rust
pub fn new(mut args: env::Args) -> Result<Config, &'static str>
```

### Step 4 - Making things clearer with `iterator adaptors`
In the search function, we had a `for` to iterate over the values.
Change it so that it is easier to understand
```rust
pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    contents
        .lines()
        .filter(|x| x.contains(query))
        .map(|x| x.trim())
        .collect()
}
```

===

# Comparing Performance
In theory, Iterators are one of the `zero-cost abstractions`, basically, it is done in a way that it does not affect performance over base way of doing things.