# Iterators
Iterators allow you to sequence a task in order - Iterating over items in a collection and know when it has ended.
```rust
let v1 = vec![1,2,3];
let v1_iter = v1.iter();
for v in v1_iter { println!("Got: {}", v);};
```
Basically a `for_each`, makes it less repetitive and a bit better in some cases!

## Iterator Trait and `next` Method
```rust
pub trait Iterator {
    type Item;
    fn next(&mut self) -> Option<Self::Item>; // Define it in your type
    // methods with default implementations elided
}
```
`type Item` and `Self::Item` means that the type that implements `Iterator` must implement a node for holding the items it has. (Chapter 19)

`next` changes the current interator into the next one in the collection
```rust
assert_eq!(v1_iter.next(), Some(&1));
assert_eq!(v1_iter.next(), Some(&2));
assert_eq!(v1_iter.next(), Some(&3));
assert_eq!(v1_iter.next(), None);
```
`iter()` transforms it into an immutable iterator, the values we get form the iterator will be immutable, for example the `.next()` will be another immutable iterator.
`into_iter()` takes ownership
`iter_mut()` is `iter()` but mutable.

## Methods that consume the iterator
When something calls `next()` which changes which value the iterator is at, it is called a *consuming adaptor*.
Consuming it means ownership is lost.
```rust
let total: i32 = v1_iter.sum();
assert_eq!(total, 6);
// We can't use v1_iter anymore, sum() consumed it
```

## Methods that produce Iterators
*iterator adaptors* are methods that change an iterator into another kind.
They are *lazy* methods, so that they don't do anything until the iterator is consumed or used.
```rust
// Ex: use map(self, Closure) that performs a closure on each iterator
let v1: Vec<i32> = vec![1, 2, 3];
//v1.iter().map(|x| x + 1);// Does not compile, iterator not consumed.
let v2: Vec<_> = v1.iter().map(|x| x+1).collect(); // Consumes iterators into a collection
assert_eq!(v2, vec![2,3,4]);
```
The closure uses the iterator value and changes it.

### Closures that capture their environment
`filter(self, closure) -> bool`
On True, value is included in the iterator, on false, value is not included
That means that `filter().collect()` creates a new collection with some items only.

Use example in `lib.rs`:
```rust
fn shoes_in_size(shoes: Vec<Shoe>, shoe_size: u32) -> Vec<Shoe> {
    shoes.into_iter().filter(|s| s.size == shoe_size).collect()
}
```
- `shoes_in_size` fun takes a vector of shoes, a size to filter, and calls an iterator that collects shoes with the size passed.

## Taking our own Iterators
```rust
impl Iterator for Struct {
    type Item = type;
    fn next(&mut self) -> Option<Self::Item> {
        //... what to do
            Some(self.item)
        //...else
            None
    }
}
```
In `what to do` we write the condition for next to return something or what to return as well as advance the iterator into the next iterator of the collection.

After implementing `next()` all other functions are already available.