# Closures
A closure is an anonymous function that can save a variable and pass as arguments to other functions.

## Closure Abstraction

### App Example
Ex: App that generates a workout given some user information
It takes a couple seconds and we only want to execute it when the user asks for a new workout.
```rust
use std::thread;
use std::time::Duration;

fn simulated_expensive_calculation(intensity: u32) -> u32 {
    println!("calculating slowly...");
    thread::sleep(Duration::from_secs(2));
    intensity
}
```

Then the main, which accepts the intensity of the workout and a rng that chooses from workout plans:
```rust
fn main() {
    let simulated_user_specified_value = 10;
    let simulated_random_number = 7;

    generate_workout(simulated_user_specified_value, simulated_random_number);
}
```

Then the function that generates the workout, which makes heavy use of the simulation:
```rust
fn generate_workout(intensity: u32, random_number: u32) {
    if intensity < 25 {
        println!(
            "Today, do {} pushups!",
            simulated_expensive_calculation(intensity)
        );
        println!(
            "Next, do {} situps!",
            simulated_expensive_calculation(intensity)
        );
    } else {
        if random_number == 3 {
            println!("Take a break today! Remember to stay hydrated!");
        } else {
            println!(
                "Today, run for {} minutes!",
                simulated_expensive_calculation(intensity)
            );
        }
    }
}
```
#### Refactoring with Closures
Instead of calling `simulated_expensive_calculcation` before each if lock, we define the closure and store the value.

`|closure_arguments| { function_body; return_var }`
```rust
let closure = |arg1| {
    function_body();
    return_val
};
```
This is creating like a function pointer or a static function, assigned to a variable. `closure` is now a variable that holds the definition of the closure function.

Then swap the function call with closure call
This is because we are using code that only pertains to a specific function call and makes it cleaner, so that other functions do not call it and have to modify it and tarnish the idea of that function.

#### Closure Type Inference and Annotation
 - No need to set the type of parameters (but can specify arg and return type if needed)
 - Short and relevant to a very small scope, not an arbitrary scenario
 - If unspecified, Type is inferred by first use of the closure

### Storing Closures
Using generics basically
```rust
struct Cacher<T>
where
    T: Fn(u32) -> u32,
{
    calculcation: T,
    value: Option<u32>
}
```
This structs specifies that `<T>` is any function with a single parameter `u32` that returns a `u32`.
`Fn` is a trait that functions and closures can hold
 - `Fn`, `FnMut`, `FnOnce` are similar traits (explained shortly after)

The struct has a function of calculation and its result, and this would be its implementation:
```rust
impl<T> Cacher<T>
    where T: Fn(u32) -> u32,
{
    fn new(calculation: T) -> Cacher<T> {
        Cacher {calculation, value: None}
    }

    fn value(&mut self, arg:u32) -> u32 {
        match self.value {
            Some(v) => v,
            None => {
                let v = (self.calculation)(arg);
                self.value = Some(v);
                v
            }
        }
    }
}
```
Struct is initialized through `new` based on the function/closure we give it.
When we ask the struct for the value, we check if the value is already set, and return if so or else we do the calculation again.

Then instead we chagne the closure variable, to a variable that holds a `Cacher` that implements the closure we did before.

#### Pitfalls

##### Multiple Chached Values
This implementation is not that good, it caches a single value. If we wanted to get a value with another parameter we could not.
Instead we should hold a hash map and check if values exist.
```rust
struct Cacher<T>
where
    T: Fn(u32) -> u32,
{
    calculation: T,
    value: HashMap<u32, u32>,
}

impl<T> Cacher<T>
    where T: Fn(u32) -> u32,
{
    fn new(calculation: T) -> Cacher<T> {
        Cacher {calculation, value: HashMap::new()}
    }

    fn value(&mut self, arg:u32) -> u32 {
        if let Some(ret) = self.value.get(&arg) { 
            return *ret
        }
        else {
            let v = (self.calculation)(arg);
            self.value.insert(arg, v);
            v
        }
    }
}
```
The value function checks if the value is found, else it does the calculations and caches the value.

##### Multiple Types
```rust
struct Cacher<T,R>
where
    T: Fn(&R) -> R,
{
    calculation: T,
    value: HashMap<R, R>,
}

impl<T, R> Cacher<T, R>
    where 
        T: Fn(&R) -> R,
        R: std::cmp::Eq + std::hash::Hash + Clone,
{
    fn new(calculation: T) -> Cacher<T, R> {
        Cacher {calculation, value: HashMap::new()}
    }

    fn value(&mut self, arg:&R) -> R {
        if let Some(ret) = self.value.get(arg) { 
            return ret.clone()
        }
        else {
            let v = (self.calculation)(&arg);
            if let Some(ret) = self.value.insert(arg.clone(), v.clone()) {
                ret
            } else { v.clone() }
        }
    }
}
```
That was harder.
You must specify a 2nd Generic `<R>` and add a `where` clause that it implements the `std::cmp:Eq`, `std::hash::Hash` and `Copy` traits, so that it can be used directly in replacement.
Instead of copy, we could use `Clone` but then it requires to make sure we are not moving the ownership of pointers when calling certain functions.
So I changed the closure function to get an immutable borrow instead of a copy or clone, then cloned the input.


## Using the environment
Closures can use variables in the same scope. That does not work with functors/inline functions
```rust
fn main() {
    let x = 4;

    let equal_to_x = |z| z == x; // This compiles

    fn equal_to_x_fn(z: i32) -> bool { z==x}; // This does not compile

    let y = 4;

    assert!(equal_to_x(y));
}
```
Closures allocate more memory for the variables in the scope it will use. That does not happen with functions. How do they use the variabels in the scope:
 - `FnOnce`: Takes ownership of the variables, moves them in a single instance per variable
 - `FnMut`: Borrows the values mutably
 - `Fn`: Borrows the values immutably
 - `move`: Moves the ownership at definition and does not let it go until it goes out of scope
