use std::error::Error;
use std::fs;
use std::env;

pub fn run(config: Config) -> Result<(), Box<dyn Error>> { // Take ownership of Config (is it a copy?)
    let contents = fs::read_to_string(config.filename)?; // Needs move capabilities to use the filename

    let results = if config.case_sensitive {
        search(&config.query, &contents)
    } else {
        search_case_insensitive(&config.query, &contents)
    };

    for line in results {
        println!("{}", line);
    }

    Ok(())
}

// Use specific struct to demark intent on the usage of arguments
#[derive(Debug)]
pub struct Config {
    pub query: String,
    pub filename: String,
    pub case_sensitive: bool,
}

impl Config {
    // Function receives a borrowed array/vector of String
    // Returns a Result, with Ok(Config) that holds intent, or error message
    pub fn new(mut args: env::Args) -> Result<Config, &'static str> {
        args.next(); // Skip 1st argument
        let query = match args.next() {
            Some(arg) => arg,
            None => return Err("Missing arg1"),
        };
        let filename = match args.next() {
            Some(arg) => arg,
            None => return Err("Missing arg2"),
        };

        println!("{:?}", env::var("CASE_INSENSITIVE"));
        let case_sensitive = env::var("CASE_INSENSITIVE").is_err();
        Ok(Config { query, filename, case_sensitive})
    }
}

pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    contents
        .lines()
        .filter(|x| x.contains(query))
        .map(|x| x.trim())
        .collect()
}

pub fn search_case_insensitive<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    let up_query: String = query.to_uppercase(); // tp_uppercase Allocates to Heap a new String!

    contents
        .lines()
        .filter(|x| x.to_uppercase().contains(&up_query))
        .map(|x| x.trim())
        .collect()
}
//==================================================================
// Tests
#[cfg(test)]
mod tests {
    use super::*;
    
    #[test]
    fn one_result() {
        let query = "duct";
        let contents = "\
            Rust: 
            safe, fast, productive. 
            Pick three.";

        assert_eq!(vec!["safe, fast, productive."], search(query, contents));
    }

    #[test]
    fn case_insensitive() {
        let query = "RuSt";
        let contents = "\
        Rust: 
        safe, fast, productive. 
        Pick three.
        Trust me.";

        assert_eq!(vec!["Rust:", "Trust me."], search_case_insensitive(query, contents));
    }
}