# Bringing Paths/Modules into Scope with `use`
`use` is the same as `using` in C++ to bring a namespace, but with a step less of involement;
```rust
// C++
using namespace std;
// We get direct access to the funcitonality inside `std` without having to say `**std::**functionality`

// Rust
use crate::mod1;
// To use funcionality in `mod1` we need to say `mod1::functionality`
// I like it more, it is like bringing specific functionality to front amd still have clear way of knowing which is being used.
```

Usually it is done this way for namespaces with functions, but to use structs we do the full step of specifying the struct we want to use
```rust
use std::collections;
use std::collections::HashMap;
//...
let mut map = collections::HashMap::new(); // This is valid
let mut map_s = HashMap::New(); // This is better and more widely used for structs
```
But that depends on if different variables have name collisions

## Solving Name Collisions - `as` Keyword
```rust
std::fmt::Result;
//std::io::Result; // This one collides
std::io::Result as IoResult; // use as to give it a new name to avoid collision
```

## Re-exporting names with `pub` `use`

```rust
mod front_of_house {
    pub mod hosting {
        pub fn add_to_waitlist() {}
    }
}
/*
    hosting would be usable by front_of_house::hosting
    we could also do `use crate::front_of_house::hosting`
        then do `hosting::`
    
    However, binaries/libraries that use this code, will not have access as front_of_house is only visible to this scope.

    We could do `pub mod front_of_house`, or `pub use crate::front_of_house::hosting` somewhere else
    Now we get `hosting::` functionality in the scope and out of the scope when other binaries/libs do `use crate::`
    
    Maybe we want users to acces a submodule but not the whole module and make it easier to get there.
*/
pub use crate::front_of_house::hosting;

pub fn eat_at_restaurant() {
    hosting::add_to_waitlist();
    hosting::add_to_waitlist();
    hosting::add_to_waitlist();
}
```

## Using External Packages

Add the package with a semantic version in `Cargo.toml`:
`rand = "0.8.4"`

Then in the core use it:
`use rand::<some_submodule>`
Rand is already accessible when added as a create dependency!

## Nested Paths for `use`
If we want to use multiple items in the same create:
```rust
use std::cmp::Ordering;
use std::io;
// or
use std::{
    cmp::Ordering,
    io,
}; // Can be made single line

//====
use std::io;
use std::io::Write;
// or
use std::io{self, Write};
```

## Glob Operator
If you want to bring ALL PUBLIC items in the scope
`use std::collections::*`