# Validating References with Lifetimes

Every reference/borrow has a lifetime, the scope in which is valid. Most of the time it goes by scope and is inferred.

Rust has a strict lifetime check - The Borrow Checker:
```rust
// A
let r;
{
    let x = 5;
    r = &x; // Invalid, x lifetime will end and R will still hold the reference to it
}
```

## Generic Lifetimes in Functions
```rust
fn longest(str1: &str, str2: &str) -> &str {
    if(str1.len() > str2.len())
    {
        str1
    }
    else {
        str2
    }
}
```
In the following function, we don't explicitly know if str1 or str2 is returned. Thus we can't know which reference should extend its lifetime for the return -> Borrow Checker Lifetime error.

## Lifetime Annotation Syntax
```rust 
&i32        // a reference
&'a i32     // a reference with an explicit lifetime
&'a mut i32 // a mutable reference with an explicit lifetime
```
This syntax specifies a "standard" lifetime for a reference. It is made to specify that multiple variables or return values have the same lifetime, so that the borrow checker does allow its usage.

```rust
fn longest<'a>(str1: &'a str, str2: &'a str) -> &'a str {
    if(str1.len() > str2.len())
    {
        str1
    }
    else {
        str2
    }
}

// In function, we specify a lifetime, to which str1, str2 and return value adhere to and will promise to keep.
```
It does not specify a `value` of lifetime, it just makes sure that the variables will at least have the `same` lifetime.
```rust
let string1 = String::from("long string is long");
let string2 = String::from("xyz");
let result1 = longest(string1.as_str(), string3.as_str());

let result2;
{
    let string3 = String::from("xyz");
    result2 = longest(string1.as_str(), string3.as_str());
}
println!("The longest string is {}", result, result2);
// result1 will be valid, as both string1 and string2 share the same lifetime
// result2 will not be, as string3 has shorter lifestime, but we use a result which will be dropped after the scope as it will get the lifetime of the lowest lifetime of the variables passed
```

# Thinking in terms of Lifetimes
```rust
fn longest<'a>(x: &'a str, y: &str) -> &'a str {
    x
}
```
In this funciton, we only care about `x` lifetime, as we don't return evern `y`. We can only specify for `x` then.

# Lifetime annotations in Struct
Structs can hold references, but require a lifetime specified:
```rust
struct ImportantExcerpt<'a> {
    part: &'a str,
}
```
This means, that at struct creation time, `part` will be of the same lifetime of the variable is borrowing from. `ImportantExcerpt` is not allowed outlive `part`.

# Lifetime Ellision
Some patterns require the use of lifetime by default, which are now picked up by the compiler and can be not specified as they would just make sense to add.

There are a set rules that apply to `fn` and `impl` blocks, to determine if a lifetime specification can be implicit.

## Rule 1 - Input only
if each parameter would have a unique lifetime.
```rust
fn foo<'a>(x: &'a i32);
fn foo<'a, 'b>(x: &'a i32, y: &'b i32);
//...
```
These can be elluded

## Rule 2
Just 1 input with lifetime parameter, it will also be applied to the output parameter
```rust
fn foo<'a>(x: &'a i32) -> &'a i32;
```

## Rule 3 
Multiple lifetimes, but one is `&self` or `&mut self`. This makes it a method, thus the default lifetime will be the one of `self` which is the needed for the method to work!

## Ellision in Action
```rust
fn first_word(s: &str) -> &str {} // Code Input

// 1st rule
fn first_word<'a>(s: &'a str) -> &str {} // Compiler Output

//2nd Rule
fn first_word(s: &str) -> &'a str {} // Compiler Output
```

Now with the `longest` fn:
```rust
fn longest(str1: &str, str2: &str) -> &str{};
// 1st Step
fn longest<'a, 'b>(str1: &'a str, str2: &'b str) -> &str {}
// It specify a unique lifetime to each parameter
// We can't check 2nd rule, as compiler can't know which of the 2 variables will matter for the return value
// 3rd rule does not apply because we don't have `& / &mut self`
```

## Lifetime Annotation in Methods
Use the same syntax as with `<T>` Generics. 
```rust
impl<'a> ImportantExcerpt<'a> {
    fn level(&self) -> i32 { 3 }

    fn announce_and_return_part(&self, announcement: &str) -> &str {
        println!("Attention please: {}", announcement);
        self.part
    }
}
```
`level` applies 1st Rule, unique lifetime per character
`announce_and_return` applies 3rd Rule, `&self` determines lifetime and return gets the same lifetime as `&self`

# The Static Lifetime
`'static` means that the variable can live for the duration of the whole program. String literals `str` have that lifetime by default.

Don't use it to solve problems, solve them correctly, they will create a ton of memory and they affect performance.

===

# All EXAMPLE
```rust
use std::fmt::Display;

fn longest_with_an_announcement<'a, T>(
    x: &'a str,
    y: &'a str,
    ann: T,
) -> &'a str
where
    T: Display,
{
    println!("Announcement! {}", ann);
    if x.len() > y.len() {
        x
    } else {
        y
    }
}
```

`longest` string function from before, but with announcements. We use generic `<T>` to detemine that we get a `<T>` that implements `Display` Trait.

Then we specify that both `x` and `y` have the same `'a` lifetime as the return and can be used to be return values.