# Traits: Shared Behaviour
Traits are a declaration (Signature) of behaviour. Something that implements a trait must implement a set of functionality to satisfy holding that trait:
```rust
pub trait Summary {
    fn summarize(&self) -> String; // Trait holder must implement method summarize, which returns a String
}
```

## Implementing a Trait
```rust
pub struct NewsArticle {
    pub headline: String,
    pub location: String,
    pub author: String,
    pub content: String,
}

impl Summary for NewsArticle {
    fn summarize(&self) -> String {
        format!("{}, by {} ({})", self.headline, self.author, self.location)
    }
}
```
`impl TRAIT_NAME for Type` Then implement trait specific methods!

You can implement:
 - External Traits on Internal Types -> Add functionality to your crate
 - Internal Traits on External Types -> For users of the crate

You can't implement
 - External Traits on External Types -> Would add unknow functionality to the external type, breaking code you are not anywhere near author of.

Traits can also define methods, not just declare them. Those can also depend on other methods and declarations. The type that implements a `Trait` has the obligation of implementing the declared but undefined methods.

## Traits as Parameters
```rust
pub fn notify(item: &impl Summary) {
    println!("Breaking news! {}", item.summarize());
}
```
This is the basic form to tell: `Any type that implements Summary`

### Trait Bound Syntax
```rust
pub fn notify<T: Summary>(item: &T) {
    println!("Breaking news! {}", item.summarize());
}
```
More verbose version for just 1 type, but more clear:
` Any <T>pye that implements Summary`
```rust
pub fn notify(item1: &impl Summary, item2: &impl Summary){//...
// Can have multiple types

pub fn notify<T: Summary>(item1: &T, itm2: &T){//...
// Clearer but single type
```

### Multiple Traits
```rust
pub fn notify(item1: &impl Summary + Display){//...

pub fn notify<T: Summary + Display>(item1: &T){//...
```
Use `+` syntax to use multiple traits.

### `where` Clauses - Clearer Traits
```rust
fn some_function<T: Display + Clone, U: Clone + Debug>(t: &T, u: &U) -> i32 {//...
// So long, quite unclear

fn some_function<T, U>(t: &T, u: &U) -> i32
    where T: Display + Clone,
          U: Clone + Debug
{//...
// Better, much clearer
```
Use the `where` syntax for declaring the traits implemented

### Return type that implements trait
In chapter 17 :D

## Solving Largest Function Issue
```rust
fn largest_t<T: PartialOrd + Copy>(list: &[T]) -> T {
    let mut largest = list[0];
    for &item in list {
        if item > largest {
            largest = item;
        }
    }
    largest
}
```
We add `PartialOrd` type as a requirement, to solve the `>` usage.
Then, we are doing a copy at `let mut largest = list[0]`, which by default would be a move, which is not legal as we are borrowing list, not owning it.
By adding `Copy` trait, now the variable passed must implement `Copy` and that action will become a copy instead of a move.

But `Copy` is stack only, so sometimes this won't be good enough for most types. `Clone` type does the copy but in Heap which would make that operation legal on more types as data is duplicated and then ownership is set too. 

The issue is also that the function returns owning `T`. Instead of returning the full value, we could return a reference to an element of the list -> Index or `&T`.

## Blanket Implementatin -> Trait Bounds to Conditionally Implement Methods
Basically you can define the base function, and then redefine it in the implementation for the type.
Also that you can make a type that implements something, automatically implement a specific type:

```rust
impl<T: Display> ToString for T {//...
```

Anything that implements `Display` will implement the `ToString` trait too. In the implementation of the `ToString` type you can use the `Display` methods!

Then you can create implementations based on the subimplemented types:
```rust
impl<T> Pair<T> {
    fn new(x: T, y: T) -> Self {
        Self { x, y }
    }
}

impl<T: Display + PartialOrd> Pair<T> {
    fn cmp_display(&self) {
        if self.x >= self.y {
            println!("The largest member is x = {}", self.x);
        } else {
            println!("The largest member is y = {}", self.y);
        }
    }
}
```
Some `<T>` that implements `Display + PartialOrd` will implement `new` and also `cmp_display`. But base will not implement `cmp_display`.