# Removing Duplication - Extracting a function
If you have an operation that could be done multiple times, extract it into a function:
```rust
// Ex: function_deduplication
fn largest(list: &[i32]) -> i32 {
    let mut largest = list[0];

    for &item in list {
        if item > largest {
            largest = item;
        }
    }

    largest
}

fn main() {
    let number_list = vec![34, 50, 25, 100, 65];

    let result = largest(&number_list); // 1st Instance
    println!("The largest number is {}", result);

    let number_list = vec![102, 34, 6000, 89, 54, 2, 43, 8];

    let result = largest(&number_list);// 2nd Instance, worth it to make into function
    println!("The largest number is {}", result);
}
```

===

# Generic Data Types

Can be used anywhere. Previous function only worked for `i32` types but could work for any that implements a `> <` comparator.
So we could change it with:
```rust
fn largest<T>(list: &[T]) -> T {
    //...
}
```
Now it accepts any type by specifying it. But this would not compile, because it does not know if `<T>` implements a comparison trait (`std::cmp::PartialOrd`), explained further on this chapter.

## Struct Definitions
```rust
struct Point<T> {
    x: T,
    y: T,
}
```
Simple, now the x and y depend on the type passed, can be both integer or float,...
But that must be consistent in the definition of a variable:
```rust
let good = Point { x: 5, y: 4}; // Integer 100%
let good2 = Point { x: 5.0f64, y 4.0f64}; // Float64 100%, can specify type after value
let bad = Point { x: 5, y:4.0}; // Ambiguous, first declares as integer, 2nd then changes to float
```
If you want different types, you can specify multiple generics in the definition:
```rust
struct Point<T,U> {
    x: T,
    y: U,
}
```
Now all combinations are valid for point.

## Enum Definitions
Remember `Option<T>`, that's it, the same can be done for `enums` as in `structs`

```rust
enum Result<T,E> {
    Ok(T),
    Err(E),
}
```

## Method Definitions
```rust
impl<T> Point<T> {
    fn x(&self) -> &T { &self.x }
}
```
Add `<Generic, Types,...>` after `impl`, if you wan the methods to be Generic

```rust
impl Point<f32> {
    fn distance from_origin(&self) -> f32 { 
        (self.x.powui(2) + self.y.powi(2)).sqrt()
    }
}
```
Or specify implementation for specific types!

## Performance of Generics
No performance cost -> It compiles into type specific versions USED, while code looks clean and Generic.