# Common Collections
Collections are data structures, which Rust provides through its standard library.

Collections store data on the heap.

# Vectors
``` rust
let v: Vec<i32> = Vec::new();
```
If we are not initializing it based on a list of values, you have to specify the type like any other type of variable:
`let v = vec![1,2,3];` 
Using the macro `vec!` creates a `Vector` from the list of items you give it, using the initial type as the type for the new vector.

## Updating a Vector
```rust
let mut v = Vec::new(); // No type inferred
v.push(<item>); // Add item to the back // Type is now inferred
```

## Dropping Vector = Dropping Vector's Heap
When a vector goes out of scope, 

## Reading Elements of a Vector
`[index]` direct access or `.get(index)` using match expression:
```rust
let third_val = &v[2];
let var = v.get(2);
if let Some(var2) = var {
    // Do something, var is an Option with Some(value of v[2]) or None
};
```
When having access or referencing a value, we can't actually modify the vector, we are holding a reference to it so ownership applies!

## Iterating over the Values in a Vector
We can use the loops we learned previously:
```rust
let mut v = vec![100,32,57];
for i in &mut v{
    *i += 50; // Dereferencing to modify the value
}
```

## Storing Multiple types in an Enum
Enums are defined as a single type, but each value in an enum can hold different data.
Then we can define a Vector that uses an Enum Type as its Generic, then we can use the enum values to store different types of values inside the Vector!
```rust
enum SpreadsheetCell{
    Int(i32),
    Float(f64),
    Text(String),
}

let row = vec![
    SpreadsheetCell::Int(3),
    SpreadsheetCell::Text(String::from("blue")),
    SpreadsheetCell::Float(10.12),
]
```
This is similar to a `void*` vector, each type is a pointer to a certain type of data in heap in different positions, but the vector holding these pointers is compact in heap.

===

# Strings
Collection of bytes, with methods to interpret them as text.

`str` is th string slice of a part of the programs binary data.
`String` type is a growable UTF-8 string type.
There are other types (`OsString`, `OsStr`, `CString`, `Cstr`), as well as other crates providing that functionality. 

## New String
`let mut s = String::new();`
Now we can add things to the string.

```rust
let data = "initial contents"
let s = data.to_string(); // Available on types that implement `Display` trait
let s = "initial contents".to_string(); // Same but avoid 1 stack allocation
let s = String::from("initial contents"); // Creat from any string, which does not care about language and characters, it encodes to UTF-8
```

## Updating a String
```rust
let mut s = String::from("foo");

// Append
s.push_str("bar");
s.push_str(s1); // Append another string, which adds the string slice, does not take ownership
s.push('a'); // Append a single character
```

```rust
// Concatenation
let s2 = s1 + &s;

```
 - s1 is invalidated as it is moved from s1 to s2
 - &s requires `&` because of the `+` operator (in-depth later)
 - + adds 2 `str`, so `&s` becomes `&s[..]` which dereferences it into a string slice (Chapter 15), wihtout taking into ownership
Using the `+` operator produces a bit of unclear behaviour, takes ownership of 1st var and gets a reference an copies into the 1st var, then returns ownership of that together.

We can use formatting to a better effect with strings:
```rust
// Concatenation with formatting
let s = format!("{}-{}-{}", "tic", "tac", "toe");
```

## Indexing in a String - Internal Representaion and How "characters" are encoded
We can't use `[]` in a `String` type, like in string slices.
Although internally it is a `Vec<u8>`, but a character may occupy more than 8bits, like in the cyrillic alphabet!

A `String` holds charaters, Scalar Values and Grapheme Clusters, a string that may appear to have 13 "letters" may well have a length double that, as those "letters" are actual characters that require multiple values to be described.

Then how do we get characters -> for with automatic slicing
```rust
for c in "नमस्ते".chars() {
    println!("{}", c); // Prints each character in succession
}
```

This works well for constant byte agroupations but is not a correct way of handling all strings. Not provided in the standard library but there are `crates` available.

**So, Strings are not simple**

===

# Hash Maps
Instead of using an index, you use a key generated by a function
This key can be anything you want, but it then generates an "index" based on the key which will access a specific value.

## New Hash Map
```rust
use std::collections::HashMap;
let mut scores = HashMap::new();

scores.insert(String::from("Blue"), 10);
scores.insert(String::from("Yellow"), 50);

// Form 2 - Join 2 vectors of different types
let teams = vec![String::from("Blue"), String::from("Yellow")];
let initial_scores = vec![10, 50];

let mut scores: HashMap<_, _> =
    teams.into_iter().zip(initial_scores.into_iter()).collect();
```
`<_ , _>` infers the typoe from the collect operation
`.zip` joins into a tuple the iterator passed into it.
`collect()` which turns it into an item for a hash map.


## Ownership in Hash Maps
`Copy` types are copied, `Move` types are moved, Hash Map always gets ownership. If you use a hash map of references, then the items which are referenced must be valid for as logn as the HashMap is valid too.

## Accessing values in HashMap
`hashmap.get(&key_var)` returns an `Option`
We can also iterate using a tuple of variabels to get them
```rust
for(key, value) in &hashmap {
    println!("{}:{}", key, value);
}
```

## Updating a Hashmap
```rust
// just insert into the same key
scores.insert(String::from("Blue"), 10);
scores.insert(String::from("Blue"), 20);

// Insert if it does not exist
scores.entry(String::form("Blue")).or_insert(50);
// Entry returns another Option, if the entry exists it will not execute as the value already is set, else it will return mutable reference to an empty value to be filled
```

### Updating based on old value
```rust
    use std::collections::HashMap;
    let text = "hello world wonderful world";
    let mut map = HashMap::new();

    for word in text.split_whitespace() {
        let count = map.entry(word).or_insert(0); //If the word was not in the map, the entry is created and set to 0

        // Count becomes the reference to the value keeping count fo words
        
        *count += 1; // We dereference and +1 the count of that word (value in keymap).   
    }
    println!("{:?}", map);
```

## Hashing Functions
The default function is called `SipHash` which is not the fastest but a quite secure system.

That can be changed specifying a `BuildHasher` train in another function/struct/... (Chapter 10), be it by yourself or from a crate