# Managing Growing Projects with Packages, Crates and Modules
**Packages**: A Cargo feature that lets you build, test, and share crates
**Crates**: A tree of modules that produces a library or executable
**Modules** and use: Let you control the organization, scope, and privacy of paths
**Paths**: A way of naming an item, such as a struct, function, or module

# Packages and Crates

## Crate
Binary or Library
`Create root` is where the code start running from in that binary

## Package
A Combination of 1 library and many binaries, but must contain at least 1 crate

## Ex:
```bash
$ cargo new my-project
     Created binary (application) `my-project` package
$ ls my-project
Cargo.toml
src
$ ls my-project/src
main.rs
```
Creatign a new project generates a set of files, `Cargo.toml` indicates the package base.
A `crate` is a collection of files, suhc as the ones in src `main.rs`. By default the build system just uses the base files it has, so it had a `lib.rs` it would contain a library instead of a binary?

Crates are to join functionality together (so a crate is like a library called in C/C++).

# Defining Modules to Control Scope and Privacy
Each binary and library may be under different scopes/namespaces, as well as have a different set view, public or private.

```rust
mod module_name {
    mod submodule_name {
        fn fun1(){}
        //...
    }
    //...
    struct struct_name {}
    const var: type = asdasd;
    enum enum_name {
        Enum1,
    }
    // Traits??
}
```
Basically a `module` is a namespace.
And this is hierarchical, to a base `module` that would be the `crate`.

# Paths for Referring to an item in the Module Tree

If we want to use things in the module, call it with `::`
Ex: `front_of_house::serving::take_order()`
Ex2: Absolute path = `**crate**::front_of_house::serving::take_order()`

`crate` or crate_name is used to know which binary/library we are using (in this case the same one we are at so `crate` directly is valid).

But if we want something to be seen out of the library, it has to be marked with `pub`, as everything is private by default even in the same file

Privacy is relative to hierarchy. Child members have knowledge of their parents, but parents do not allow their child to be publicly viewed by default.

For childs to use things outside their scope, start with `super` which is like `../` in path terminology.

## Making Structs and Enums public

`pub` before a structs makes it public, but not its fields... have ot add `pub` to each field...

Enum become all public when using `pub` before them.

# Bringing Paths/Modules into Scope with `use`
`use` is the same as `using` in C++ to bring a namespace, but with a step less of involement;
```rust
// C++
using namespace std;
// We get direct access to the funcitonality inside `std` without having to say `**std::**functionality`

// Rust
use crate::mod1;
// To use funcionality in `mod1` we need to say `mod1::functionality`
// I like it more, it is like bringing specific functionality to front amd still have clear way of knowing which is being used.
```

Usually it is done this way for namespaces with functions, but to use structs we do the full step of specifying the struct we want to use
```rust
use std::collections;
use std::collections::HashMap;
//...
let mut map = collections::HashMap::new(); // This is valid
let mut map_s = HashMap::New(); // This is better and more widely used for structs
```
But that depends on if different variables have name collisions

## Solving Name Collisions - `as` Keyword
```rust
std::fmt::Result;
//std::io::Result; // This one collides
std::io::Result as IoResult; // use as to give it a new name to avoid collision
```

## Re-exporting names with `pub` `use`

```rust
mod front_of_house {
    pub mod hosting {
        pub fn add_to_waitlist() {}
    }
}
/*
    hosting would be usable by front_of_house::hosting
    we could also do `use crate::front_of_house::hosting`
        then do `hosting::`
    
    However, binaries/libraries that use this code, will not have access as front_of_house is only visible to this scope.

    We could do `pub mod front_of_house`, or `pub use crate::front_of_house::hosting` somewhere else
    Now we get `hosting::` functionality in the scope and out of the scope when other binaries/libs do `use crate::`
    
    Maybe we want users to acces a submodule but not the whole module and make it easier to get there.
*/
pub use crate::front_of_house::hosting;

pub fn eat_at_restaurant() {
    hosting::add_to_waitlist();
    hosting::add_to_waitlist();
    hosting::add_to_waitlist();
}
```

## Using External Packages

Add the package with a semantic version in `Cargo.toml`:
`rand = "0.8.4"`

Then in the core use it:
`use rand::<some_submodule>`
Rand is already accessible when added as a create dependency!

## Nested Paths for `use`
If we want to use multiple items in the same create:
```rust
use std::cmp::Ordering;
use std::io;
// or
use std::{
    cmp::Ordering,
    io,
}; // Can be made single line

//====
use std::io;
use std::io::Write;
// or
use std::io{self, Write};
```

## Glob Operator
If you want to bring ALL PUBLIC items in the scope
`use std::collections::*`

===

# Separating modules in different files

Modules can be declared, rather than defined/ Then we define the module in a file that has the same name:
```rust
// src/lib.rs
mod front_of_house;

pub use crate::front_of_house::hosting;

pub fn eat_at_restaurant() {
    hosting::add_to_waitlist();
    hosting::add_to_waitlist();
    hosting::add_to_waitlist();
}

// src/front_of_house.rs
pub mod hosting; // The file acts as a "crate" as said before

// src/front_of_house/hosting.rs 
pub fn add_to_waitlist() {}

// hosting.rs is set in a folder that "acts" like the crate holding the definitions made in front_of_house.rs
// the .rs file acts as the "namespace" directly!
```