# Implementing an Object-Oriented Design Pattern
The *state pattern* says that there is some internal state, represented by *state objects* which determine the behaviour of the program.

Each *state* determines which is the next one and how to behave.

## Example: Blog post workflow

1. A blog post starts as an empty draft.
2. When the draft is done, a review of the post is requested.
3. When the post is approved, it gets published.
4. Only published blog posts return content to print, so unapproved posts can’t accidentally be published.

This steps go sequentially, we can't publish a blog post that is not even in review.
```rust
use blog::Post;

fn main() {
    let mut post = Post::new(); // New Draft

    post.add_text("I ate a salad for lunch today"); // Modify Draft
    assert_eq!("", post.content()); // Post will still be empty, we have not yet posted the acualt post

    post.request_review(); // Draft sent to Review
    assert_eq!("", post.content()); // Post still empty

    post.approve(); // Draft Approved, Post is now set with its content
    assert_eq!("I ate a salad for lunch today", post.content()); // Yey
}
```

### Defining Post
```rust
// Post holds a Trait Obejct that implements a State
// Held in an Option<T> because 
pub struct Post {
    state: Option<Box<dyn State>>, // Holds something that can have a State, private so we control what can be created here!
    content: String, // The Official post that is/will be on the blog, private so that the contents can only be changed through correct means
}

impl Post {
    pub fn new() -> Post { // Generate a new Draft the intended way!
        Post {
            state: Some(Box::new(Draft {})),
            content: String::new(),
        }
    }

    pub fn add_text(&mut self, text: &str) {
        self.content.push_str(text);
    }

    pub fn content(&self) -> &str {
        self.state.as_ref().unwrap().content(self)
        // This will call the content method of `State`
        // If there are `Virutal` sub implementation, that will be called
    }

    // Public function to request review of a `Post`
    pub fn request_review(&mut self) {
        // take() obtains the value from Some which State is
        if let Some(s) = self.state.take() { 
            self.state = Some(s.request_review())
        }
    }

    // Public implementation for approve of a `Post`
    pub fn approve(&mut self) {
        if let Some(s) = self.state.take() {
            self.state = Some(s.approve())
        }
    }
}

trait State {
    // Private function to change a State
    // Only valid for Box<Self> types, and takes ownership as it is intended for the function to change the state!
    
    // Draft will change into PendingReview!
    fn request_review(self: Box<Self>) -> Box<dyn State>;

    // Delegated to be reimplemented, PendingReview will change to Published!
    fn approve(self: Box<Self>) -> Box<dyn State>;

    // Only Publish will reimplement this method
    // Other states will not return the post contents
    fn content<'a>(&self, post: &'a Post) -> &'a str {
        ""
    }
}

// Empty struct to reference a type that is just implementation based
struct Draft {}

impl State for Draft {
    // Private implementation that does the state change
    fn request_review(self: Box<Self>) -> Box<dyn State> {
        Box::new(PendingReview {})
    }

    fn approve(self: Box<Self>) -> Box<dyn State> {
        self
    }
}

// Empty ""
struct PendingReview {}

impl State for PendingReview {
    fn request_review(self: Box<Self>) -> Box<dyn State> {
        self
    }

    fn approve(self: Box<Self>) -> Box<dyn State> {
        Box::new(Published {})
    }
}

// Empty ""
struct Published {}

impl State for Published {
    fn request_review(self: Box<Self>) -> Box<dyn State> {
        self
    }

    fn approve(self: Box<Self>) -> Box<dyn State> {
        self
    }

    // This sub implementation will send the string only when the post is ready to be published!
    fn content<'a>(&self, post: &'a Post) -> &'a str {
        &post.content
    }
}
```

## Try the exercises
Here are the new blog post flow, the implementation of only to change text on Draft was very painful, does not look greate and a lot of `mut` trown around to get it to be used.

 - Add a reject method that changes the post’s state from PendingReview back to Draft.
 - Require two calls to approve before the state can be changed to Published.
 - Allow users to add text content only when a post is in the Draft state. Hint: have the state object responsible for what might change about the content but not responsible for modifying the Post.
```rust
// Post holds a Trait Obejct that implements a State
// Held in an Option<T> because 
pub struct Post {
    state: Option<Box<dyn State>>, // Holds something that can have a State, private so we control what can be created here!
    content: String, // The Official post that is/will be on the blog, private so that the contents can only be changed through correct means
    approvals: u32,
}

impl Post {
    pub fn new() -> Post { // Generate a new Draft the intended way!
        Post {
            state: Some(Box::new(Draft {})),
            content: String::new(),
            approvals: 0,
        }
    }

    pub fn add_text(&mut self, text: &str) {
        //let mut edit_text: &String = self.state.as_ref().unwrap().add_text(self);
        //println!("{:?}", self.state.as_ref().unwrap().add_text(self).take());
        if let Some(s) = self.state.as_ref().unwrap().add_text(& mut self.content).take() {
            println!("Tried adding text");
            s.push_str(text);
        }
    }

    pub fn content(&self) -> &str {
        self.state.as_ref().unwrap().content(self)
        // This will call the content method of `State`
        // If there are `Virutal` sub implementation, that will be called
    }

    // Public function to request review of a `Post`
    pub fn request_review(&mut self) {
        // take() obtains the value from Some which State is
        if let Some(s) = self.state.take() { 
            self.state = Some(s.request_review())
        }
    }

    // Public implementation for approve of a `Post`
    pub fn approve(&mut self) {
        if let Some(s) = self.state.take() {
            self.approvals += 1;
            if(self.approvals > 1){
                self.state = Some(s.approve())
            }
            else {
                self.state = Some(s)
            }
        }
    }
}

trait State {
    // Private function to change a State
    // Only valid for Box<Self> types, and takes ownership as it is intended for the function to change the state!
    fn request_review(self: Box<Self>) -> Box<dyn State>;
    
    // Delegated to be reimplemented, PendingReview will change to Published!
    fn approve(self: Box<Self>) -> Box<dyn State>;

    // Only Publish will reimplement this method
    // Other states will not return the post contents
    fn content<'a>(&self, post: &'a Post) -> &'a str {
        ""
    }

    fn add_text<'a>(&self, post: &'a mut String) -> Option<&'a mut String> {
        None
    }

    fn reject(self: Box<Self>) -> Box<dyn State>;
}

// Empty struct to reference a type that is just implementation based
struct Draft {}

impl State for Draft {
    // Private implementation that does the state change
    fn request_review(self: Box<Self>) -> Box<dyn State> {
        Box::new(PendingReview {})
    }

    fn approve(self: Box<Self>) -> Box<dyn State> {
        self
    }

    fn reject(self: Box<Self>) -> Box<dyn State> {
        self
    }

    fn add_text<'a>(&self, post: &'a mut String) -> Option<&'a mut String> {
        Some(post)
    }
}

// Empty ""
struct PendingReview {}

impl State for PendingReview {
    fn request_review(self: Box<Self>) -> Box<dyn State> {
        self
    }

    fn approve(self: Box<Self>) -> Box<dyn State> {
        Box::new(Published {})
    }

    fn reject(self: Box<Self>) -> Box<dyn State> {
        Box::new(Draft {})
    }
}

// Empty ""
struct Published {}

impl State for Published {
    fn request_review(self: Box<Self>) -> Box<dyn State> {
        self
    }

    fn approve(self: Box<Self>) -> Box<dyn State> {
        self
    }

    // This sub implementation will send the string only when the post is ready to be published!
    fn content<'a>(&self, post: &'a Post) -> &'a str {
        &post.content
    }

    fn reject(self: Box<Self>) -> Box<dyn State> {
        self
    }
}
```



## Trade-Offs
It is a cool pattern, easy to extend in some ways, a bit more exhaustive in others. Adding new steps really only makes you implement a function in each step, and implementing the new step `struct`.

There is some coupling between steps, but it is still not much work to replace the couplings.

But, there is a lot of duplication, we see that we have to implement the `request_review` and `approve` method for each step, because the `Trait` object does not allow to return `self`, only when it is known at compile time which is in the Struct definition, not in the trait definition.