# Pattern Syntax

Recap on all patterns seen:

## Matching Literals
```rust
let x = 1;

match x {
    1 => println!("one"),
    2 => println!("two"),
    3 => println!("three"),
    _ => println!("anything"),
}
```

## Matching Named Variables
Named shadow variables in a match, will shadow variables (redefine locally the variables with the same name at parent scope) of the same name that are defined in the scope!
```rust
let x = Some(5);
let y = 10;

match x {
    Some(50) => println!("Got 50"),
    Some(y) => println!("Matched, y = {:?}", y), // y becomes 5, not 10!
    _ => println!("Default case, x = {:?}", x),
}
println!("at the end: x = {:?}, y = {:?}", x, y); // y is 10 again
```
To compare outer values, we will need `Match Guard Conditionals`, explaine later on.

## Multiple Patters
We can use `|` (OR) to join conditions for a match!
```rust
let x = 1;
match x{
    1 | 2 => println!("one or two"),
    3 => println!("three"),
    _ => println!("anything"),
}
```

## Matching Ranges with values `..=`
Using the syntax `a..b` or `a..=b` we can match values that are contained in a range!
```rust
let x = 5;
match x {
    1..=5 => println!("1 to 5, included 5"),
    6..10 => println!("6 to 10, not included 10"),
    'a'..='z' => println!("Syntax also works on chars!"),
    _ => println!("anything"),
}
```

## Destructuring - Breaking Apart Values

### Destructuring `Struct`
We can break apart the values of struct into own variables:
```rust
struct Point {
    x: i32,
    y: i32,
}
//...
let p = Point { x:0, y: 7};
let Point {x:a, y:b} = p; // Break into differently named variables
println!("{} and {}", a, b);

//...
let Point {x,y} = p; // Break into same name variables
```

You can also use a match pattern with sepcific values on a Struct!
```rust
let p = Point {x:0, y:7};
match p {
    Point { x, y:0} => println!("On the X axis at {}", y), 
    // if y==0, we don't care about x
    Point { x:0, y} => println!("On the Y axis at {}", x),
    // if x==0, we don't care about y
    Point { x, y } => println!("On neither axisL ({},{})", x,y),
    // We don't care about any value, just use the, (like '_' in other match statements, a catchall)
}
```

### Destructuring `Enum`
Used previously on match patters for `Result<T,E>` and `Option<T>`, we have to expect for all enum types and their variables:
```rust
enum Message {
    Quit,
    Move { x: i32, y: i32 },
    Write(String),
    ChangeColor(i32, i32, i32),
}

fn main() {
    let msg = Message::ChangeColor(0, 160, 255);

    match msg {
        Message::Quit => {
            println!("The Quit variant has no data to destructure.")
        }
        Message::Move { x, y } => {
            println!(
                "Move in the x direction {} and in the y direction {}",
                x, y
            );
        }
        Message::Write(text) => println!("Text message: {}", text),
        Message::ChangeColor(r, g, b) => println!(
            "Change the color to red {}, green {}, and blue {}",
            r, g, b
        ),
    }
}
```
We can't check against specific types of an enum subtype:
```rust
match msg {
//...
    Message::ChangeColor(r:0, g,b) => println!("This is illegal"),
    Message::ChangeColor => println!("Also Illegal, must use match the type to destructure!"),
//...
}
```

### Destructuring Nexted `Struct` and `Enum`
```rust
enum Color {
    Rgb(i32, i32, i32),
    Hsv(i32, i32, i32),
}

enum Message {
    Quit,
    Move { x: i32, y: i32 },
    Write(String),
    ChangeColor(Color),
}

fn main() {
    let msg = Message::ChangeColor(Color::Hsv(0, 160, 255));

    match msg {
        Message::ChangeColor(Color::Rgb(r, g, b)) => println!(
            "Change the color to red {}, green {}, and blue {}",
            r, g, b
        ),
        Message::ChangeColor(Color::Hsv(h, s, v)) => println!(
            "Change the color to hue {}, saturation {}, and value {}",
            h, s, v
        ),
        Message::ChangeColor(color) => match color { // Nest another match also valid
            Color::Rgb(r,g,b) => println!("Valid!"),
            Color::Hsv(h,s,v) => println!("Valid!"),
        },
        Message2::ChangeColor(color) => {}, // Can get value, but still need to match against subtype to properly use
        _ => (),
    }
}
```

### Destructuring `Struct` and `Tuple`
Like with any other type of variable, you have to follow the type to destructure it:
```rust
struct Point {
    x:i32,
    y:i32,
}
enum SomeType {
    Type1(((i32, i32), Point)),
}
//...
let ((feet, inches,), Point{x,y}) = ((10,10), Point{x:10, y:10});
```
Weirdly, this does not seem that useful, the example provided does not seem actually useful.
Tried:
```rust
let t1 = SomeType::Type1((10,10), Point{x:10, y:10});
let ((feet, inches,), Point{x,y}) = t1; // Invalid, requires matching to get the `Type1`

let t2 = ((10,10), Point{x:10, y:10});
let ((feet, inches,), Point{x,y}) = t2; // Valid!
```
Seems worth to get return values of functions in a single value, then destructure them when actually needed.


## Ignoring Values in a Pattern
When you don't actually need/want to complete the match statement, with `_`

### Ignoring entire values with `_`
Can be used in ANY pattern!
```rust
fn foo(_:i32, y:i32) { println!("We only care about y)");} //Why?
let Point {x:_, y} = p; // we ignore a part of the variable
let (_, y) = (10,10);
```

### Ignoring parts of a value with a Nested `_`
```rust
let mut setting_value = Some(5);
let new_setting_value = Some(10);

match (setting_value, new_setting_value) {
    (Some(_), Some(_)) => {
        println!("Can't overwrite an existing customized value");
    }
    _ => {
        setting_value = new_setting_value;
    }
}

println!("setting is {:?}", setting_value);
//...

let numbers = (2, 4, 8, 16, 32);

match numbers {
    (first, _, third, _, fifth) => {
        println!("Some numbers: {}, {}, {}", first, third, fifth)
    }
}
```

### Ignoring unused values
When you don't use a value that you have declared, the compiler will warn about not having used it, with `_varname` it will not warn:
```rust
let _x = 5; //_x is still used, and binds as variable `_x`
let y = 10; // Compiler warns about y not being used
//...
let s = Some(String::("Hello!"));
if let Some(_s) = s { println!("var is bound to _s, then dropped");}
println!("{:?}", s); // illegal, moved to _s then dropped
//...
if let Some(_) = s {} // Does not bind to anything, does not drop it
```

### Ignoring Remaining parts with `..`
```rust
let v = (1,2,3,4,5);
let (x,..) = v; // We ignore the rest
let (x,.., second_last, last) = v; // Also valid, we ignore inbetween
let (.., last) = v; // Also valid, we ignore the start

let (.., inbetween_?, ..) = v; // Invalid, we don't know how many are to get the inbetween. More than 1 `..` is not allowed
```

## Extra Conditionals with `match guards`
`match guard` is an additional `if` that makes use of the value matched:
```rust
let num = Some(4);
match num {
    Some(x) if x < 5 => println!("Less than five: {}", x),
    Some(x) => pritnln!("{}", x), // Another to catch all x values
    None => (),
}
```
This way we can get some value and compare it to another outer value

You can use multiple checks in a pattern using the OR operator `|`:
```rust
// Example: | and multiple conditions
let num = Some(5);
let y = true;
match num {
    Some(4 | 5 | 6) if y == true => (),
    //...
}
```

## `@` Bindings
`@` operator, allows to test a bound variable:
```rust
match num {
    Point{x: px @ 4..6, y} => println!("it works!"),
    //...
}
```
This syntax only seems to work with struct type tho