# Using Trait Objects that Allow for Values of Different Types

In Chapter 8, we used:
```rust
enum SpreadsheetCell{
    Int(i32),
    Float(f64),
    Text(String),
}
//...
let row = vec![
    SpreadsheetCell::Int(3),
    SpreadsheetCell::Text(String::from("blue")),
    SpreadsheetCell::Float(10.12),
]
```
This is a valid solution to make a `Vector` that holds multiple type, KNOWN AT COMPILE TIME.

But what if the user might extend the types at runtime? Or the type can't be known to the library?

## What is a Trait Object
Something that implements a Trait, but with unknown type such as:
 - `Box<dyn Debug>` : We know that we need an object that implements `Debug` trait, but we don't care or know what type it is!

## Example: GUI code interface
In GUIs, the user will inevitably create new items which the library might not specifically know about. 
An abstract class would have `draw()` method that worked on every type of GUI `Component`.

### Defining Common Behaviour - Trait
`trait` objects are basically a commonground to declare some functionality. They are not like other languages' objects.

A basic implementation might be like this, with the Screen allowing `ANY` object that implements `Draw`.
```rust
// As before, every GUI item should implment a draw functionality
pub trait Draw {
    fn draw(&self);
}

// Screen is basically where things are draw, so it holds all the items to be drawn
pub struct Screen {
    pub components: Vec<Box<dyn Draw>>,
}

// When you run the program, and run the components in scree, they will be drawn
impl Screen {
    pub fun run(&self) {
        for component in slef.components.iter() {
            component.draw();
        }
    }
}
```

Should this be done with Generics, that would not be possible:
```rust
// Screen only accepts a SINGLE type, that MUST implement Draw
// Ex: Screen might only be able to hold `Buttons`
pub struct Screen<T: Draw> {
    pub components: Vec<T>,
}

// Same here, this would work for the struct of a single type
impl<T> Screen<T>
where
    T: Draw,
{
    pub fn run(&self) {
        for component in self.components.iter() {
            component.draw();
        }
    }
}
```

### Implementing the Trait
```rust
pub struct Button { 
    pub width: u32,
    pub height: u32,
    pub label: String,
}

impl Draw for Button {
    fn draw(&self) {
        // draw button code...
    }
}
```
This type now implements Draw. If the user implemented this type, the library would be able to handle it when using `Trait` objects as it successfully implements the `Draw` trait. With Generics, that would work too but now the library would only handle `Buttons`, not any other type.

*duck typing* - If it walks like a duck and quaks like a duck, must be a duck - If something behaves as you expect to, you should treat it as you expect the type to behave.

```rust
// Button implements `Draw`, this will work
let screen = Screen {
    components: vec![Box::new(Button::new()),...],
};


// String does not, it will not compile!
let screen_bad = Screen {
    components: vec![Box::new(String::from("Where is Draw?"))],
};
```

## Dynamic Dispatch for Trait Objects
With Generics, the compiler generates the specific code for the types that used it - *static dispatch*

With Traits, that does not occurr, the compiler does not know the types that might use that trait object. That cost is paid at runtime, where instead of inherently knowing what that code would do, it has to reroute to what the specific implementation is.
Similar to a pointer to a child definition of an inherited function: Parent Virtual Fun -> Child Defined Fun

### Trait Object Safety Requirements
Methods on a `Trait` Object have to:
 - Not return `Self` ever, because that requires guarantee of the knowledge of Type
 - No Generic Type Parameters, because they will be tied to a specific type

If the methods don't follow this principles, they will not be eligible to be used as `Trait` Objects.

For example, `Clone` implements the method `clone()` which returns a copy of the value it is called by. It requires a return owning the copied value of `Self` thus not being safe.