# Fearless Concurrency
 - Concurrency -> Parts of a program execute independently
 - Parallelism -> Parts of a program execute at the same time
It is hard - Not in Rust

# Using Threads to Run Code Simultaneously
Program = 1 Process
System can execute multiple processes
Why can't program execute multiple processes then? They can, but is more complex:
 - Race Conditions : Multiple threads accessing the same data
 - DeadLocks : Threads waiting on each other to finish using a resource
 - Esoteric and hard to replicate bugs...

Thread creation differ per language:
 - OS Threads - 1:1 Threads -> 1 Thread created in language = 1 Operating System thread that executes
 - Green Threads - M:N Threads -> Any number of threads created in Language, execute in an arbitrary uncontrolled number of OS Threads. Also known as fibers, as threads will be more versatile.

Rust implements 1:1 by itself, because it relies on being "small" in binary size - its runtime (the library that is compiled minimally to allow the code to run) is very small.

## Creating new Threads - `spawn`
```rust
use std::thread;
use std::time::Duration;

fn main() {
    // Create 1 thread that executes the following closure
    thread::spawn(|| {
        for i in 1..10 {
            println!("hi number {} from the spawned thread!", i);
            thread::sleep(Duration::from_millis(1));
        }
    });

    // Thread spawn ends and main continues executing
    for i in 1..5 {
        println!("hi number {} from the main thread!", i);
        thread::sleep(Duration::from_millis(1));
    }
}
// When execution of main ends, spawned thread is also killed as it is dependent on it
```

### Waiting for threads to end - `join`
```rust
let thread_handle = thread::spawn(|| {...});
handle.join().unwrap();
// Join blocks current thread andd waits for the thread to end its execution
// Join returns a Result<T> we could handle with a match expression, unwrap_or_else,... for this we let it panic with unwrap on error.
```
`join()` blocks current thread, then waits for the other thread to end its execution.
It returns a `Result<T>`

If called before the `main thread loop`, then it will print all the lines form the `spawned thread loop` before executing anything, as it is blocked waiting fo `spawned` to finish executing.

### Using Move closures with threads
Previously, we were able to move values into a closure that was being executed. That was because we were not changing the ownership of the data into another thread:
```rust
let v = vec![1, 2, 3];
let handle = thread::spawn(|| {
    println!("Here's a vector: {:?}", v);
});
    // Imagine we dropped V here!
    // std::mem::drop(v);
    // What if thread had not yet tried accessing V?
handle.join().unwrap();
```
`println!` requires reference to `v`, so the closure borrows it. But the lifetime of the thread is unknown the the borrow, thus the lifetime of the borrow can't be inferred -> Invalid.

We can add `move` before the closure to force the ownership to transfer, now it will not be valid in the main thread.

===

# Using Message Passing to Transfer Data between Threads
`Message Passing` on thread/actor send messages containing data - "Share memory by communicating, not communicating by sharing data."

`channel` - Implmentation of communicating service. 1 Transmitter and 1 Receiver. Transmitter sends, receiver gets. If one is dropped, the channel is dropped.
```rust
use std::sync::mpsc;
// Multiple Producer, Single Consumer
fn channel_test() {
    // Generate transmitter-receiver pair in a tuple
    // Use let (tuple) to destructure into 2 variables!
    let (tx,rx) = mpsc::channel();
    // By itself, this does not compile
    // We send the transmitter into another thread
    thread::spawn(move || { 
        let val = String::from("Hi from spawned!");
        tx.send(val).unwrap();
        // send(...) return `Result<T,E>` if the receiver is dropped, returns error
    });

    let received = rx.recv().unwrap();
    println!("Got message: {}", received);
}
```
`recv()` blocks the thread until it receives something, returning a `Result<T,E>`. If the transmitter were to be dropped, it would return error.

`try_recv()` does not block and returns an immediate `Result<T,E>`, with `Err` if there are no messages to receive.

## Channels and Ownership transfer
`send()` can transfer ownership of values from transmitter to receiver, a value that has been sent will no longer be accessible:
```rust
thread::spawn(move || {
    let val = String::from("hi");
    tx.send(val).unwrap();
    println!("val is {}", val); // Illegal, main has val, not spawned
});
```

## Sending Multiple Values and check on the Receiver
```rust
thread::spawn(move || {
    let vals = vec![
        String::from("hi"),
        String::from("from"),
        String::from("the"),
        String::from("thread"),
    ];

    for val in vals {
        tx.send(val).unwrap();
        thread::sleep(Duration::from_secs(1));
    }
});

for received in rx {
    println!("Got: {}", received);
}
```
The iterator `foreach` will now become like a for loop waiting on `recv()`, each time a new message is sent, an iteration takes place in main.

### Multiple Producers - Cloning the Transmitter
We can actually just `clone()` the transmitter and send that clone to another thread. The receive will receive from both without issues.
```rust
let tx1 = tx.clone();
thread::spawn(move || {
    let vals = vec![
            String::from("more"),
            String::from("messages"),
            String::from("for"),
            String::from("you"),
        ];

        for val in vals {
            tx.send(val).unwrap();
            thread::sleep(Duration::from_secs(1));
        }
});
```
Here we spawn another thread sending more messages at the same time. Remeber that ordering is a non-guarantee when receiving form multiple sources, we can't know exactly execution times, process prioritazion,...  Have to be careful!

===

# Shared-State Concurrency
In previous phrase "Not communicating by sharing memroy" : Multiple ownership model. Multiple sources can access memory at the same time = COMPLEX but also effective.

## Mutexes - One Thread at a Time
`mutex` = mutual exclusion, only 1 exclusive accessor. You have to ask permission to the `mutex`'s lock to access data.
 - MUST attemp to acquire the lock before using data (ask for permission)
 - When you're done, express so other can use it
Basically like borrowing a book from a library.

## API of `Mutex<T>`
```rust
fn mutex() {
    let m = Mutex::new(5); // Create a mutex holding a var
    {
        let mut num = m.lock().unwrap(); 
        // Ask for permission - Blocks execution
        // The call will fail if another thread holds the lock and fucks up, so this thread to treat that case.
        // Returns a `smart pointer` type `MutexGuard` that implements the `Deref` trait so we can get to the value with ownership.
        *num = 6;

        // `MutexGuard` also implements `Drop` so the lock becomes available when dropped
    }
    println!("m = {:?}", m);
}
```

### Sharing a Mutex<T> between Threads
```rust
let counter = Mutex::new(0);
let mut handles = vec![];

// This will happen 10 times
// We move ownership of both handle and counter on the 1st iteration
// Next iterations can't move, because it is not main who owns them, its the 1st iteration that owns them...
for _ in 0..10 {
    let handle = thread::spawn(move || {
        let mut num = counter.lock().unwrap();

        *num += 1;
    });
    handles.push(handle);
}

for handle in handles {
    handle.join().unwrap();
}

println!("Result: {}", *counter.lock().unwrap());
```
### Multiple Ownership in Multiple Threads
We can try and us `Rc<T>`:
```rust
use std::rc::Rc;
fn mutex_multiple_owners_Rc_2() {
    let counter = Rc::new(Mutex::new(0));
    let mut handles = vec![];

    for _ in 0..10 {
        let counter = Rc::clone(&counter); // Rc is not thread safe 
        let handle = thread::spawn(move || {
            let mut num = counter.lock().unwrap();

            *num += 1;
        });
        handles.push(handle);
    }

    for handle in handles {
        handle.join().unwrap();
    }

    println!("Result: {}", *counter.lock().unwrap());
}
```

We get this error:
```
error[E0277]: `Rc<Mutex<i32>>` cannot be sent between threads safely
   --> src/main.rs:11:22
    |
11  |           let handle = thread::spawn(move || {
    |  ______________________^^^^^^^^^^^^^_-
    | |                      |
    | |                      `Rc<Mutex<i32>>` cannot be sent between threads safely
12  | |             let mut num = counter.lock().unwrap();
13  | |
14  | |             *num += 1;
15  | |         });
    | |_________- within this `[closure@src/main.rs:11:36: 15:10]`
    |
    = help: within `[closure@src/main.rs:11:36: 15:10]`, the trait `Send` is not implemented for `Rc<Mutex<i32>>`
    = note: required because it appears within the type `[closure@src/main.rs:11:36: 15:10]`

For more information about this error, try `rustc --explain E0277`.
error: could not compile `shared-state` due to previous error
```
In short, `Rc<T>` is not safe in multithreaded environments, it does not implement the `Send` trait which is used to pass around info between threads/channels.
Why is it not safe? Because each call to `clone()` and `drop()` does not use any safety measures to make sure data is synced. Imagine 2 threads call `clone()` at the same time, One would increment `4->5` and the other do the same `4->5`, but we have increased 2 times. Then with drops the same could happen or even worse, the value caused to be dropped early and another thread use it!

### Atomic Reference Counting - `Arc<T>`
Basically thread-safe `Rc<T>`
Atomics are types that can only be accessed 1 at a time, guaranteed by the language and OS architechture <https://doc.rust-lang.org/std/sync/atomic/index.html>

But that 1 at a time, has a cost of waiting, so trade safety for performance, be smart when using them.
```rust
use std::sync::Arc;

fn mutex_multiple_owners_Arc_3() {
    let counter = Arc::new(Mutex::new(0));
    let mut handles = vec![];

    for _ in 0..10 {
        let counter_in = Arc::clone(&counter); 
        // We borrow counter intro loop just to clone it
        // Move each clone into every thread
        // 1 clone per thread of the Mutex<Counter>
        // Make it atomic so actions to it are also safe.
        let handle = thread::spawn(move || {
            let mut num = counter_in.lock().unwrap();

            *num += 1;
        });
        handles.push(handle);
    }

    for handle in handles {
        handle.join().unwrap();
    }

    println!("Result: {}", *counter.lock().unwrap());
}
```

## Similarities - RefCell/Rc vs Mutex/Arc
Basically they are very similar, `Mutex<T>` allows getting a mutable reference from an immutable variable (Interior Mutability) - Like `Cell` Types.

Same as explained with `Rc/Arc`.

`Mutex` are not safe by default, we can create `deadlock` which is when 2 or more threads lock a resource each, then try to acquire the lock of the resource the other has locked:
```rust
fn deadlock() {
    let t1_var = Arc::new(Mutex::new(0));
    let t2_var = Arc::new(Mutex::new(0));
    let t1_clone = t1_var.clone();
    let t2_clone = t2_var.clone();

    let handle1 = thread::spawn(move || {
        let mut t1 = t1_var.lock().unwrap();
        thread::sleep(Duration::from_secs(2));
        println!("Got t1 lock, try getting t2 lock...");
        let mut t2 = t2_var.lock().unwrap();
        println!("This never prints...");
    });

    let handle2 = thread::spawn(move || {
        let mut t2 = t2_clone.lock().unwrap();
        thread::sleep(Duration::from_secs(2));
        println!("Got t2 lock, try getting t1 lock...");
        let mut t1 = t1_clone.lock().unwrap();
        println!("This never prints...");
    });
    println!("Threads Spawned...");
    handle1.join().unwrap();
    handle2.join().unwrap();
}
```
This code will lock each thread a variable, then wait a couple seconds to make sure the other thread locked the other.
Then try to lock the one the other thread has -> Deadlock.

===

# Extending Concurrency with `Sync` and `Send` traits
The Rust language has very few `concurrency` specific features. They are part of the standard library not language. You can still create your own primitives and features.

These are `std::marker` traits, meaning, that implementing them does not require implementing other methods, it jsut marks them as safe for the operations they entail.

## Allowing Transference of Ownership between Threads with `Send`
Remember we used `send(val)` to send a complete value and ownership was transferred, that implements `Send` trait.

## Allowing Access from Multiple Threads with `Sync`
The `Sync` marker trait, tells that it is safe for the type to be referenced from multiple threads.

## Actually Implementing `Send` and `Sync` - UNSAFE
Types that are made of `Send` and `Sync` types, become automatically `Sync` and `Send` safe. They don't have to be implemented manually.

Actually implementing the type on one that uses non-safe `Send/Sync` types, requires that the methods and functions implemented on the type, are manually made to be safe in the multithreaded environments, which most likely involves you using unsafe code.

More on the [Rustonomoicon](https://doc.rust-lang.org/nomicon/index.html)