# IO Projcet: Grep
Command Line project, that uses skills in the previous chapters!
`GREP` -> *Globally* search a *Regular* *Expression* and *Print*
Take a file + strign argument -> return lines that contain that string and print them

## Objectives
 - Read Environment Variables
 - Print Error Messages to console (`stderr` `stdout`)
 - Use of terminal features
 - Consolidate Chapters 7->11
 - Introduce closures, iterators and trait objects (CHapter 13 & 17)

===

# Accepting Command Line Arguments

`cargo run` will run the program or `/path/to/executable/exec_name.exe`
After the executing line, anything we pass separated by spaces, will be treated as arguments:
`cargo run arg1 arg2`

## Reading Arguments
Passing arguments will not yield anything by default, you have to accept them!
`std::env::args` -> function that returns an iterator of the arguments that were given to the program, as well as the execution path relative to where the run command was thrown
```rust
let args = std::env::args(); // Iterator of the arguments passed to program
```

Will panic if the arguments have invelid Unicode -> Use `std::env::args_os` if you know you have invalid Unicode, produces `OsString` but varies from platform to platform.

`Iterators` will be covered in chapter 13, they are a series of ordered values (a node in a list for example), can use `collect` method to create a collection out of an iterator.
```rust
let arg_vec: Vec<String> = args.collect();
```
## Saving the Arguments into Variables
We can have them into the vector, but when they pile up they will be harder to understand.

Use immutable borrowing for this variables, you are reading variables you pass you don't want to change them usually... but you could still borrow it mutably if needed.

===

# Reading a file
```rust
use std::fs; // Work with filesystem to load write files and other operations
//...
let file_contents = fs::read_to_string(filename) // load file and dump contents into String (not str)
    .expect("Error message"); // In case of error, panic, could be handled better
```

Not much to add, the argument is passed into the `fs::read_to_string` thus loading if the path to it is correct (accepts relative and full paths).

===

# Organizing Code, Modularity and Error Handling
It is bad that everything is in main.
`expect` is not very good, panic and bad default error message

## Separation of concerns
Create a `lib.rs` and send logic when it gets too big to be readable in main.
Main should only:
 - Call parsing logic with argument mvalues
 - Setup a configuration
 - Call a `run` function from lib.rs
 - Handle error returned by `run` if needed
`Main` runs the program, but `lib` does the logic

### Step 1 - Parse in a different function
```rust
// Function receives a borrowed array/vector of String
// Returns a tuple of 2 string slices which are the arguments
fn parse_args(args: &[String]) -> (&str, &str) { 
    // Will panic if no arguments are passed
    let query = args[1].clone();
    let filename = args[2].clone();
    (query, filename)
}
```

### Step 2 - Declare Intent with the variables
```rust
// Use specific struct to demark intent on the usage of arguments
struct Config {
    query: String,
    filename: String,
}
// Returns a Config struct, demarcating intent of usage of the extracted arguments
fn parse_args(args: &[String]) -> Config { 
    // Will panic if no arguments are passed
    let query = args[1].clone();
    let filename = args[2].clone();
    Config { query, filename}
}
```
Why clone? Because of simplicity, we could do:
```rust
struct Config<'a> {
    query: &'a str,
    filename: &'a str,
}
fn parse_args(args: &[String]) -> Config  { 
    // Will panic if no arguments are passed
    let query = &args[1];
    let filename = &args[2];
    Config { query, filename}
}
```
This would be correct, return `Config` now bound by the lifetime of the `args` passed, and internal borrows are bound by the lifetime of `Config`, but would be a bit harder to understand.

Still, remember that `clone` is slow, we are delving into the heap, it will be costly in the longrun.

### Step 3 - Create a Constructor for Config
```rust
impl Config {
    fn new(args: &[String]) -> Config {
        let query = args[1].clone();
        let filename = args[2].clone();
        Config { query, filename}
    }
}

fn main() {
    //...
    let _config = Config::new(&arg_vec);
    //...
}
```
It is better than having do everything everytime we want to create a `Config`, as well as tidier and easier to understand.


## Fixing Error Handling
Currently, program panics if we pass less that 2 args.

### Step 1 - Better Error message
Check for issues in the `new` method:
```rust
fn new(args: &[String]) -> Config {
    if args.len() < 3 {
        panic!("Note enough args");
    }
    //...
}
// Or
fn new(args: &[String]) -> Result(Config, Err) {
    if args.len() < 3{
        return Err("Less than 2 arguments passed")
    }
    let query = args[1].clone();
    let filename = args[2].clone();
    Ok(Config { query, filename})
}
```

### Step 2 - Handling Errors - new returnign Result
```rust
use std::process;
//...
let config = Config::new(&args).unwrap_or_else(|err| {
    println!("Problem parsing arguments: {}", err);
    process::exit(1); // Early exit with error number, `closure`
});
```
`unwrap_or_else` opens a scope if the return value is an `Err`, then you can call a bracket and capture the variable held inside err

`process::exit(1)` is a closure, early exit


## Extracting logic from main
Extract function that does things to data:
```rust
run(config);
//...

fn run(config: &Config) {
    let contents = fs::read_to_string(config.filename)
        .expect("Uh Oh");
    println!("Test:\n{}", contents);
}
```

### Step 1 - Improve Error Handling
```rust
use std::error::Error;

// --snip--

fn run(config: Config) -> Result<(), Box<dyn Error>> {
    let contents = fs::read_to_string(config.filename)?;
    println!("With text:\n{}", contents);
    Ok(())
}
```
Return a Result instead of allowing the `expect` `panic!`
We use the `?` operator that returns the `Err` on error, but continues on `Ok`
`Box<dyn Error>` Is basically a type that englobes any type of error (Chapter 17).

### Step 2 - Handling errors run in main
```rust
if let Err(e) = run(config) {
    println!("Application error: {}", e);

    process::exit(1);
}
// Or
run(config).unwrap_or_else(|err| {
    println!("App Error: {}", err);
    process::exit(1);
});
```
Use `if let` because we are NOT returning a variable, only on `Err`.
We could use an `unwrap_or_else` is a bit wasteful because we are forcing a return value even if we don't use it.
`if let` syntax only does error work, else it does not care (would there be any runtime differences in this case?).

## Split into a Library
Simple, just take the functionality that is not in `main`, send it to `lib.rs`, set what is needed to `pub` and tada. You can use `use mini_grep` and get everything and use it too.

===

# Developing the Lib with Test-Driven Development
Before writing the functionality:
 - Write 1 test function
 - Write functionality so that it fails
 - Refactor so that it succeeds
 - Write/Refactor fun
 - Repeat

## Test and Failing Function
Write a test that would file the intent:
```rust
// test module mumbo jumbo...
fn one_result() {
    let query = "duct";
    let contents = "\
        Rust: 
        safe, fast, productive. 
        Pick three.";

    assert_eq!(vec!["safe, fast, productive."], search(query, contents));
}

// Then the fun that should fail the test
// We tell that The borrow placed in the vectors, has the same lifetime as the string used to pass the contents, which we are borrowing from
pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    vec![]
}
```
The fun now will fail the test as it returns an empty vector!

## Make function succeed!
What does it need to do:
 - Go through each line of the text
 - Check if the line contains the query string
 - Ok -> Add to List
 - Err -> Nothing
 - Return list of lines containing it.
```rust
pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    // Create the return variable, mutable because we will add items to it
    let mut ret: Vec<&str> = Vec::new(); 
    for line in contents.lines() { //lines() converts into a collection of lines
        if line.contains(query){ //Check if query is contained
            ret.push(line.trim());// add to list if it is, trim spaces at start and end
        }
    }
    ret // Return list of lines containing query
}
```

Now it works, test with the whole program!

===

# Working with environment variables

Let's test with an Environment Variable that we will read s allowing case insensitive search!

### Step 1 - Test & Failing function
```rust
//test module...
fn case_insensitive() {
    let query = "RuSt";
    let contents = "\
    Rust: 
    safe, fast, productive. 
    Pick three.
    Trust me.";

    assert_eq!(vec!["Rust:", "Trust me."], search_case_insensitive(query, contents));
}
//...
pub fn search_case_insensitive<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    //... copy search function
}
```
This will fail because it can't find "RuSt", but could find "Rust" and "rust" which are differently cased.

### Step 2 - Making it pass
```rust
pub fn search_case_insensitive<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    let mut ret: Vec<&str> = Vec::new();
    let up_query: String = query.to_uppercase(); // tp_uppercase Allocates to Heap a new String!

    for line in contents.lines() {
        if line.to_uppercase().contains(&up_query) { // pass a reference because we are using String not str
            ret.push(line.trim());
        }
    }

    ret
}
```
Now we transform everything to uppercase (could do to lowercase too), and treat every string as that case to ignore casing!

## Adding Environment Variables
First we add the bool to check if a config is case sensitive
```rust
pub struct Config {
    pub query: String,
    pub filename: String,
    pub case_sensitive: bool,
}
```
Then we have to check if the environment variable is set:
```rust
// Config implementation
pub fn new(args: &[String]) -> Result<Config, &str> {
    //...
    let case_sensitive = env::var("CASE_INSENSITIVE").is_err();
    Ok(Config { query, filename, case_sensitive })
}
```
`is_err` Returns false if the variable is found! So bool is false because it should not be `case_sensitive` the search.

To test this out, I swapped to a powershell instead of windows cmdprompt terminal.
```bash
$Env:VarName=Value # Adds variable to current session
Remove-Item Env:VarName # Removes variables
```

===

# Writing Errors to Standard Error instead of Standard Output
`println!` writes to output terminal, but we have `stdout` and `stderr` so that users can filter.

In the command line we see everything by default, but when we output to a file, that may not be the case or what is wanted

## Printing to Standard Error
`eprintln!`, that's it, now it will print to command line but not to files output like `cargo run > output.txt`

`> file.txt` redirects output to a file, we would see on command line only the rrors but in file the output.