# Removing Duplication - Extracting a function
If you have an operation that could be done multiple times, extract it into a function:
```rust
// Ex: function_deduplication
fn largest(list: &[i32]) -> i32 {
    let mut largest = list[0];

    for &item in list {
        if item > largest {
            largest = item;
        }
    }

    largest
}

fn main() {
    let number_list = vec![34, 50, 25, 100, 65];

    let result = largest(&number_list); // 1st Instance
    println!("The largest number is {}", result);

    let number_list = vec![102, 34, 6000, 89, 54, 2, 43, 8];

    let result = largest(&number_list);// 2nd Instance, worth it to make into function
    println!("The largest number is {}", result);
}
```

===

# Generic Data Types

Can be used anywhere. Previous function only worked for `i32` types but could work for any that implements a `> <` comparator.
So we could change it with:
```rust
fn largest<T>(list: &[T]) -> T {
    //...
}
```
Now it accepts any type by specifying it. But this would not compile, because it does not know if `<T>` implements a comparison trait (`std::cmp::PartialOrd`), explained further on this chapter.

## Struct Definitions
```rust
struct Point<T> {
    x: T,
    y: T,
}
```
Simple, now the x and y depend on the type passed, can be both integer or float,...
But that must be consistent in the definition of a variable:
```rust
let good = Point { x: 5, y: 4}; // Integer 100%
let good2 = Point { x: 5.0f64, y 4.0f64}; // Float64 100%, can specify type after value
let bad = Point { x: 5, y:4.0}; // Ambiguous, first declares as integer, 2nd then changes to float
```
If you want different types, you can specify multiple generics in the definition:
```rust
struct Point<T,U> {
    x: T,
    y: U,
}
```
Now all combinations are valid for point.

## Enum Definitions
Remember `Option<T>`, that's it, the same can be done for `enums` as in `structs`

```rust
enum Result<T,E> {
    Ok(T),
    Err(E),
}
```

## Method Definitions
```rust
impl<T> Point<T> {
    fn x(&self) -> &T { &self.x }
}
```
Add `<Generic, Types,...>` after `impl`, if you wan the methods to be Generic

```rust
impl Point<f32> {
    fn distance from_origin(&self) -> f32 { 
        (self.x.powui(2) + self.y.powi(2)).sqrt()
    }
}
```
Or specify implementation for specific types!

## Performance of Generics
No performance cost -> It compiles into type specific versions USED, while code looks clean and Generic.

===

# Traits: Shared Behaviour
Traits are a declaration (Signature) of behaviour. Something that implements a trait must implement a set of functionality to satisfy holding that trait:
```rust
pub trait Summary {
    fn summarize(&self) -> String; // Trait holder must implement method summarize, which returns a String
}
```

## Implementing a Trait
```rust
pub struct NewsArticle {
    pub headline: String,
    pub location: String,
    pub author: String,
    pub content: String,
}

impl Summary for NewsArticle {
    fn summarize(&self) -> String {
        format!("{}, by {} ({})", self.headline, self.author, self.location)
    }
}
```
`impl TRAIT_NAME for Type` Then implement trait specific methods!

You can implement:
 - External Traits on Internal Types -> Add functionality to your crate
 - Internal Traits on External Types -> For users of the crate

You can't implement
 - External Traits on External Types -> Would add unknow functionality to the external type, breaking code you are not anywhere near author of.

Traits can also define methods, not just declare them. Those can also depend on other methods and declarations. The type that implements a `Trait` has the obligation of implementing the declared but undefined methods.

## Traits as Parameters
```rust
pub fn notify(item: &impl Summary) {
    println!("Breaking news! {}", item.summarize());
}
```
This is the basic form to tell: `Any type that implements Summary`

### Trait Bound Syntax
```rust
pub fn notify<T: Summary>(item: &T) {
    println!("Breaking news! {}", item.summarize());
}
```
More verbose version for just 1 type, but more clear:
` Any <T>pye that implements Summary`
```rust
pub fn notify(item1: &impl Summary, item2: &impl Summary){//...
// Can have multiple types

pub fn notify<T: Summary>(item1: &T, itm2: &T){//...
// Clearer but single type
```

### Multiple Traits
```rust
pub fn notify(item1: &impl Summary + Display){//...

pub fn notify<T: Summary + Display>(item1: &T){//...
```
Use `+` syntax to use multiple traits.

### `where` Clauses - Clearer Traits
```rust
fn some_function<T: Display + Clone, U: Clone + Debug>(t: &T, u: &U) -> i32 {//...
// So long, quite unclear

fn some_function<T, U>(t: &T, u: &U) -> i32
    where T: Display + Clone,
          U: Clone + Debug
{//...
// Better, much clearer
```
Use the `where` syntax for declaring the traits implemented

### Return type that implements trait
In chapter 17 :D

## Solving Largest Function Issue
```rust
fn largest_t<T: PartialOrd + Copy>(list: &[T]) -> T {
    let mut largest = list[0];
    for &item in list {
        if item > largest {
            largest = item;
        }
    }
    largest
}
```
We add `PartialOrd` type as a requirement, to solve the `>` usage.
Then, we are doing a copy at `let mut largest = list[0]`, which by default would be a move, which is not legal as we are borrowing list, not owning it.
By adding `Copy` trait, now the variable passed must implement `Copy` and that action will become a copy instead of a move.

But `Copy` is stack only, so sometimes this won't be good enough for most types. `Clone` type does the copy but in Heap which would make that operation legal on more types as data is duplicated and then ownership is set too. 

The issue is also that the function returns owning `T`. Instead of returning the full value, we could return a reference to an element of the list -> Index or `&T`.

## Blanket Implementatin -> Trait Bounds to Conditionally Implement Methods
Basically you can define the base function, and then redefine it in the implementation for the type.
Also that you can make a type that implements something, automatically implement a specific type:

```rust
impl<T: Display> ToString for T {//...
```

Anything that implements `Display` will implement the `ToString` trait too. In the implementation of the `ToString` type you can use the `Display` methods!

Then you can create implementations based on the subimplemented types:
```rust
impl<T> Pair<T> {
    fn new(x: T, y: T) -> Self {
        Self { x, y }
    }
}

impl<T: Display + PartialOrd> Pair<T> {
    fn cmp_display(&self) {
        if self.x >= self.y {
            println!("The largest member is x = {}", self.x);
        } else {
            println!("The largest member is y = {}", self.y);
        }
    }
}
```
Some `<T>` that implements `Display + PartialOrd` will implement `new` and also `cmp_display`. But base will not implement `cmp_display`.

===

# Validating References with Lifetimes

Every reference/borrow has a lifetime, the scope in which is valid. Most of the time it goes by scope and is inferred.

Rust has a strict lifetime check - The Borrow Checker:
```rust
// A
let r;
{
    let x = 5;
    r = &x; // Invalid, x lifetime will end and R will still hold the reference to it
}
```

## Generic Lifetimes in Functions
```rust
fn longest(str1: &str, str2: &str) -> &str {
    if(str1.len() > str2.len())
    {
        str1
    }
    else {
        str2
    }
}
```
In the following function, we don't explicitly know if str1 or str2 is returned. Thus we can't know which reference should extend its lifetime for the return -> Borrow Checker Lifetime error.

## Lifetime Annotation Syntax
```rust 
&i32        // a reference
&'a i32     // a reference with an explicit lifetime
&'a mut i32 // a mutable reference with an explicit lifetime
```
This syntax specifies a "standard" lifetime for a reference. It is made to specify that multiple variables or return values have the same lifetime, so that the borrow checker does allow its usage.

```rust
fn longest<'a>(str1: &'a str, str2: &'a str) -> &'a str {
    if(str1.len() > str2.len())
    {
        str1
    }
    else {
        str2
    }
}

// In function, we specify a lifetime, to which str1, str2 and return value adhere to and will promise to keep.
```
It does not specify a `value` of lifetime, it just makes sure that the variables will at least have the `same` lifetime.
```rust
let string1 = String::from("long string is long");
let string2 = String::from("xyz");
let result1 = longest(string1.as_str(), string3.as_str());

let result2;
{
    let string3 = String::from("xyz");
    result2 = longest(string1.as_str(), string3.as_str());
}
println!("The longest string is {}", result, result2);
// result1 will be valid, as both string1 and string2 share the same lifetime
// result2 will not be, as string3 has shorter lifestime, but we use a result which will be dropped after the scope as it will get the lifetime of the lowest lifetime of the variables passed
```

# Thinking in terms of Lifetimes
```rust
fn longest<'a>(x: &'a str, y: &str) -> &'a str {
    x
}
```
In this funciton, we only care about `x` lifetime, as we don't return evern `y`. We can only specify for `x` then.

# Lifetime annotations in Struct
Structs can hold references, but require a lifetime specified:
```rust
struct ImportantExcerpt<'a> {
    part: &'a str,
}
```
This means, that at struct creation time, `part` will be of the same lifetime of the variable is borrowing from. `ImportantExcerpt` is not allowed outlive `part`.

# Lifetime Ellision
Some patterns require the use of lifetime by default, which are now picked up by the compiler and can be not specified as they would just make sense to add.

There are a set rules that apply to `fn` and `impl` blocks, to determine if a lifetime specification can be implicit.

## Rule 1 - Input only
if each parameter would have a unique lifetime.
```rust
fn foo<'a>(x: &'a i32);
fn foo<'a, 'b>(x: &'a i32, y: &'b i32);
//...
```
These can be elluded

## Rule 2
Just 1 input with lifetime parameter, it will also be applied to the output parameter
```rust
fn foo<'a>(x: &'a i32) -> &'a i32;
```

## Rule 3 
Multiple lifetimes, but one is `&self` or `&mut self`. This makes it a method, thus the default lifetime will be the one of `self` which is the needed for the method to work!

## Ellision in Action
```rust
fn first_word(s: &str) -> &str {} // Code Input

// 1st rule
fn first_word<'a>(s: &'a str) -> &str {} // Compiler Output

//2nd Rule
fn first_word(s: &str) -> &'a str {} // Compiler Output
```

Now with the `longest` fn:
```rust
fn longest(str1: &str, str2: &str) -> &str{};
// 1st Step
fn longest<'a, 'b>(str1: &'a str, str2: &'b str) -> &str {}
// It specify a unique lifetime to each parameter
// We can't check 2nd rule, as compiler can't know which of the 2 variables will matter for the return value
// 3rd rule does not apply because we don't have `& / &mut self`
```

## Lifetime Annotation in Methods
Use the same syntax as with `<T>` Generics. 
```rust
impl<'a> ImportantExcerpt<'a> {
    fn level(&self) -> i32 { 3 }

    fn announce_and_return_part(&self, announcement: &str) -> &str {
        println!("Attention please: {}", announcement);
        self.part
    }
}
```
`level` applies 1st Rule, unique lifetime per character
`announce_and_return` applies 3rd Rule, `&self` determines lifetime and return gets the same lifetime as `&self`

# The Static Lifetime
`'static` means that the variable can live for the duration of the whole program. String literals `str` have that lifetime by default.

Don't use it to solve problems, solve them correctly, they will create a ton of memory and they affect performance.

===

# All EXAMPLE
```rust
use std::fmt::Display;

fn longest_with_an_announcement<'a, T>(
    x: &'a str,
    y: &'a str,
    ann: T,
) -> &'a str
where
    T: Display,
{
    println!("Announcement! {}", ann);
    if x.len() > y.len() {
        x
    } else {
        y
    }
}
```

`longest` string function from before, but with announcements. We use generic `<T>` to detemine that we get a `<T>` that implements `Display` Trait.

Then we specify that both `x` and `y` have the same `'a` lifetime as the return and can be used to be return values.