# Enums
Non Algebraic, data type to differentiates different types in a category.
Ex:
```rust
enum IpAddressKind{
    V4,
    V6
}
// There are 2 IP Adrress types, IPv4 IPv6, but both are IP Addresses

let four = IpAddressKind::V4;
let six = IpAddressKind::V6;
// Access through namespace
```

## Attached types

To take care of type and value, we would make a struct:
```rust
// Ew, so long, type attached enums rock
struct IpAddr {
    kind: IpAddrKind,
    address: String,
}

let home = IpAddr {
    kind: IpAddrKind::V4,
    address: String::from("127.0.0.1"),
};

let loopback = IpAddr {
    kind: IpAddrKind::V6,
    address: String::from("::1"),
};
```

But Enum values can have attached data types:
```rust
enum IpAddrKind{
    V4(String),
    V6(String)
}

let home_ipv4 = IpAddrKind::V4(String::from("127.0.0.1"));
let loopback_ipv6 = IpAddrKind::V6(String::from("::1"));
```
Which is way more concise than having to declare for example a struct to take care of which type of address an ip has.

Thus making this enum values like `shorthand` structs, but not quite.
When we create such value, each variant of the enum can have different types attached
```rust
enum IpAddrKind {
    V4(u8,u8,u8,u8),
    V6(String)
}
```
In this case, the standard library already defines this `IpAddr` enum with the structs attached `Ipv4Addr` and `Ipv6Addr` to each one.

## Methods
Enums can also have methos and functions!
```rust
impl IpAddr {
    fn call(&self){}
    fn wait() {}
}
```
--- 

# Option Enum
```rust
enum Option<T> {
    None,
    Some(T)
}
```
Basically, option can either hold a value of any type or not hold anything, but still depending on the type.
Super Important, is like checking for `NULL` or `nullptr`.
`<T>` stands for generics, that it can be of any type.
```rust
let some_number = Some(5); // Option Enum that has Some with value 5
let some_string = Some("a string"); // Option Enum that has Some with string "a string"

let absent_number: Option<i32> = None; // Option Enum of type i32 that has no value (None)

// but!
let x: i8 = 5; // Type i8 Value of 5
let y: Option<i8> = Some(5); // Option of type i8, with Some of value 5
let sum = x+y; // Error, we can't use Some to actually do something
```
To access the value encoded in the enum, we need to convert it to the type we want!

How do you do that? With a method implemented for the enum.

---

# Match Control Flow operator
As seen previously `match` is used like a switch, it compares a value and has an outcope for each possible enum option:
```rust
enum Coin {
    Penny,
    Nickel,
    Dime,
    Quarter,
}
//---
fn value_in_cents(coin: Coin) -> u8 {
    match coin {
        Coin::Penny => 1,
        Coin::Nickel => 5,
        Coin::Dime => 10,
        Coin::Quarter => 25,
    }
}
```
The match condition returns the type of enum value it is, so that then it executes the one it is intended for.

This is also the way to get the associated data to an enum value
```rust
enum Coin {
    Penny,
    Nickel,
    Dime,
    Quarter(u16),// Year of minting
}
//...
match coin {
    Coin::Quarter(year) => { // Gets the associated vars in same order
        println!("Quarter made in {}", year);
        year // return value is the year and now match will return the year
    }
}
```

## Match with Option<T>

We can operate on the value that Option has if we know the type, Ex:
```rust
fn plus_one(x: Option<i32>) -> Option<i32> {
    match x {
        None => None,
        Some(i) => Some(i + 1),
    }
}

let five = Some(5);
let six = plus_one(five);
let none = plus_one(None);
```
But in `match` we have to make sure we take into account all cases, the following would not compile:
```rust
fn plus_one(x: Option<i32>) -> Option<i32> {
    match x {
        Some(i) => Some(i + 1),
    }
}
```
It is missing hte `None` arm of the option, so it throws an error.

## Catch-All Patterns -> _Placeholder
```rust
let dice_roll = 9;
// Other / whatever name not part of the enum
match dice_roll {
    3 => add_fancy_hat(),
    7 => remove_fancy_hat(),
    other => move_player(other),
}

// _ -> Value is not used
match dice_roll {
    3 => add_fancy_hat(),
    7 => remove_fancy_hat(),
    _ => (), // Not executing code, just returning a unit value
}
```
Basically, we can add a default case which would make use of the enum value, or a `_` that would not use it.


# Concise flow control - If Let

```rust
let config_max = Some(3u8); // Some of type u8 with value 3
// Default match with _ to just take care of Some(valid value) case
match config_max {
    Some(max) => println!("Valid max {}", max),
    _ =>(),
};

// If Let flow?
if let Some(var) = config_max {
    println!("Configured max {}", var);
}
```
The if let floiw is more concise, but lets destructure it.
`if (let Some(var) = config_max) /*then*/ { /* Execute some code with *var* */}`
The Condition `(let Some(var) = config_max)` will bind temporarily the vlaue of config_max to a nonexistent `Some(T)` variable, and the vlaue of `config_max(T)` will be set onto `var`.

But, if config_max were not to be able to be bound to `Some(T)` then the conditional would fail and not execute!

Then we can use `else`/`else if` to continue the statements.