# Closures
A closure is an anonymous function that can save a variable and pass as arguments to other functions.

## Closure Abstraction

### App Example
Ex: App that generates a workout given some user information
It takes a couple seconds and we only want to execute it when the user asks for a new workout.
```rust
use std::thread;
use std::time::Duration;

fn simulated_expensive_calculation(intensity: u32) -> u32 {
    println!("calculating slowly...");
    thread::sleep(Duration::from_secs(2));
    intensity
}
```

Then the main, which accepts the intensity of the workout and a rng that chooses from workout plans:
```rust
fn main() {
    let simulated_user_specified_value = 10;
    let simulated_random_number = 7;

    generate_workout(simulated_user_specified_value, simulated_random_number);
}
```

Then the function that generates the workout, which makes heavy use of the simulation:
```rust
fn generate_workout(intensity: u32, random_number: u32) {
    if intensity < 25 {
        println!(
            "Today, do {} pushups!",
            simulated_expensive_calculation(intensity)
        );
        println!(
            "Next, do {} situps!",
            simulated_expensive_calculation(intensity)
        );
    } else {
        if random_number == 3 {
            println!("Take a break today! Remember to stay hydrated!");
        } else {
            println!(
                "Today, run for {} minutes!",
                simulated_expensive_calculation(intensity)
            );
        }
    }
}
```
#### Refactoring with Closures
Instead of calling `simulated_expensive_calculcation` before each if lock, we define the closure and store the value.

`|closure_arguments| { function_body; return_var }`
```rust
let closure = |arg1| {
    function_body();
    return_val
};
```
This is creating like a function pointer or a static function, assigned to a variable. `closure` is now a variable that holds the definition of the closure function.

Then swap the function call with closure call
This is because we are using code that only pertains to a specific function call and makes it cleaner, so that other functions do not call it and have to modify it and tarnish the idea of that function.

#### Closure Type Inference and Annotation
 - No need to set the type of parameters (but can specify arg and return type if needed)
 - Short and relevant to a very small scope, not an arbitrary scenario
 - If unspecified, Type is inferred by first use of the closure

### Storing Closures
Using generics basically
```rust
struct Cacher<T>
where
    T: Fn(u32) -> u32,
{
    calculcation: T,
    value: Option<u32>
}
```
This structs specifies that `<T>` is any function with a single parameter `u32` that returns a `u32`.
`Fn` is a trait that functions and closures can hold
 - `Fn`, `FnMut`, `FnOnce` are similar traits (explained shortly after)

The struct has a function of calculation and its result, and this would be its implementation:
```rust
impl<T> Cacher<T>
    where T: Fn(u32) -> u32,
{
    fn new(calculation: T) -> Cacher<T> {
        Cacher {calculation, value: None}
    }

    fn value(&mut self, arg:u32) -> u32 {
        match self.value {
            Some(v) => v,
            None => {
                let v = (self.calculation)(arg);
                self.value = Some(v);
                v
            }
        }
    }
}
```
Struct is initialized through `new` based on the function/closure we give it.
When we ask the struct for the value, we check if the value is already set, and return if so or else we do the calculation again.

Then instead we chagne the closure variable, to a variable that holds a `Cacher` that implements the closure we did before.

#### Pitfalls

##### Multiple Chached Values
This implementation is not that good, it caches a single value. If we wanted to get a value with another parameter we could not.
Instead we should hold a hash map and check if values exist.
```rust
struct Cacher<T>
where
    T: Fn(u32) -> u32,
{
    calculation: T,
    value: HashMap<u32, u32>,
}

impl<T> Cacher<T>
    where T: Fn(u32) -> u32,
{
    fn new(calculation: T) -> Cacher<T> {
        Cacher {calculation, value: HashMap::new()}
    }

    fn value(&mut self, arg:u32) -> u32 {
        if let Some(ret) = self.value.get(&arg) { 
            return *ret
        }
        else {
            let v = (self.calculation)(arg);
            self.value.insert(arg, v);
            v
        }
    }
}
```
The value function checks if the value is found, else it does the calculations and caches the value.

##### Multiple Types
```rust
struct Cacher<T,R>
where
    T: Fn(&R) -> R,
{
    calculation: T,
    value: HashMap<R, R>,
}

impl<T, R> Cacher<T, R>
    where 
        T: Fn(&R) -> R,
        R: std::cmp::Eq + std::hash::Hash + Clone,
{
    fn new(calculation: T) -> Cacher<T, R> {
        Cacher {calculation, value: HashMap::new()}
    }

    fn value(&mut self, arg:&R) -> R {
        if let Some(ret) = self.value.get(arg) { 
            return ret.clone()
        }
        else {
            let v = (self.calculation)(&arg);
            if let Some(ret) = self.value.insert(arg.clone(), v.clone()) {
                ret
            } else { v.clone() }
        }
    }
}
```
That was harder.
You must specify a 2nd Generic `<R>` and add a `where` clause that it implements the `std::cmp:Eq`, `std::hash::Hash` and `Copy` traits, so that it can be used directly in replacement.
Instead of copy, we could use `Clone` but then it requires to make sure we are not moving the ownership of pointers when calling certain functions.
So I changed the closure function to get an immutable borrow instead of a copy or clone, then cloned the input.


## Using the environment
Closures can use variables in the same scope. That does not work with functors/inline functions
```rust
fn main() {
    let x = 4;

    let equal_to_x = |z| z == x; // This compiles

    fn equal_to_x_fn(z: i32) -> bool { z==x}; // This does not compile

    let y = 4;

    assert!(equal_to_x(y));
}
```
Closures allocate more memory for the variables in the scope it will use. That does not happen with functions. How do they use the variabels in the scope:
 - `FnOnce`: Takes ownership of the variables, moves them in a single instance per variable
 - `FnMut`: Borrows the values mutably
 - `Fn`: Borrows the values immutably
 - `move`: Moves the ownership at definition and does not let it go until it goes out of scope

===

# Iterators
Iterators allow you to sequence a task in order - Iterating over items in a collection and know when it has ended.
```rust
let v1 = vec![1,2,3];
let v1_iter = v1.iter();
for v in v1_iter { println!("Got: {}", v);};
```
Basically a `for_each`, makes it less repetitive and a bit better in some cases!

## Iterator Trait and `next` Method
```rust
pub trait Iterator {
    type Item;
    fn next(&mut self) -> Option<Self::Item>; // Define it in your type
    // methods with default implementations elided
}
```
`type Item` and `Self::Item` means that the type that implements `Iterator` must implement a node for holding the items it has. (Chapter 19)

`next` changes the current interator into the next one in the collection
```rust
assert_eq!(v1_iter.next(), Some(&1));
assert_eq!(v1_iter.next(), Some(&2));
assert_eq!(v1_iter.next(), Some(&3));
assert_eq!(v1_iter.next(), None);
```
`iter()` transforms it into an immutable iterator, the values we get form the iterator will be immutable, for example the `.next()` will be another immutable iterator.
`into_iter()` takes ownership
`iter_mut()` is `iter()` but mutable.

## Methods that consume the iterator
When something calls `next()` which changes which value the iterator is at, it is called a *consuming adaptor*.
Consuming it means ownership is lost.
```rust
let total: i32 = v1_iter.sum();
assert_eq!(total, 6);
// We can't use v1_iter anymore, sum() consumed it
```

## Methods that produce Iterators
*iterator adaptors* are methods that change an iterator into another kind.
They are *lazy* methods, so that they don't do anything until the iterator is consumed or used.
```rust
// Ex: use map(self, Closure) that performs a closure on each iterator
let v1: Vec<i32> = vec![1, 2, 3];
//v1.iter().map(|x| x + 1);// Does not compile, iterator not consumed.
let v2: Vec<_> = v1.iter().map(|x| x+1).collect(); // Consumes iterators into a collection
assert_eq!(v2, vec![2,3,4]);
```
The closure uses the iterator value and changes it.

### Closures that capture their environment
`filter(self, closure) -> bool`
On True, value is included in the iterator, on false, value is not included
That means that `filter().collect()` creates a new collection with some items only.

Use example in `lib.rs`:
```rust
fn shoes_in_size(shoes: Vec<Shoe>, shoe_size: u32) -> Vec<Shoe> {
    shoes.into_iter().filter(|s| s.size == shoe_size).collect()
}
```
- `shoes_in_size` fun takes a vector of shoes, a size to filter, and calls an iterator that collects shoes with the size passed.

## Taking our own Iterators
```rust
impl Iterator for Struct {
    type Item = type;
    fn next(&mut self) -> Option<Self::Item> {
        //... what to do
            Some(self.item)
        //...else
            None
    }
}
```
In `what to do` we write the condition for next to return something or what to return as well as advance the iterator into the next iterator of the collection.

After implementing `next()` all other functions are already available.

===

# Improving I/O Project
Old
```rust
pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    let mut ret: Vec<&str> = Vec::new();
    for line in contents.lines() { //lines() converts into a collection of lines
        if line.contains(query){
            ret.push(line.trim());
        }
    }

    ret
}

impl Config {
    // Function receives a borrowed array/vector of String
    // Returns a Result, with Ok(Config) that holds intent, or error message
    pub fn new(args: &[String]) -> Result<Config, &str> {
        if args.len() < 3{
            return Err("Less than 2 arguments passed")
        }
        let query = args[1].clone();
        let filename = args[2].clone();
        println!("{:?}", env::var("CASE_INSENSITIVE"));
        let case_sensitive = env::var("CASE_INSENSITIVE").is_err();
        Ok(Config { query, filename, case_sensitive})
    }
}
//===============================
// main.rs
use std::env; // Not needed as std can be used directly, but would be shorter
use std::process;
use mini_grep::Config;

fn main() {
    println!("Start the program!");

    // Obtaining the arguments
    let arg_vec: Vec<String> = env::args().collect(); // Convert into a Vector of strings
    let _config = Config::new(&arg_vec).unwrap_or_else(|err| {
        println!("Error parsing arguments: {}", err);
        process::exit(1); // Stop execution with error number
    });

    println!("{:?}", _config);

    mini_grep::run(_config).unwrap_or_else(|err| {
        println!("App Error: {}", err);
        process::exit(1);
    });
}
```
### Step 1 - Use the Iterator from env::args
```rust
pub fn new(mut args: env::Args) -> Result<Config, &str>
// then main
//...
// let arg_vec: Vec<String> = env::args().collect(); // Delete this
    
let _config = Config::new(env::args()).unwrap_or_else(//...
```

### Step 2 - Iterate over values instead of indexing in
```rust
args.next(); // Skip 1st argument
let query = match args.next() {
    Some(arg) => arg,
    None => return Err("Missing arg1"),
}
let filename = match args.next() {
    Some(arg) => arg,
    None => return Err("Missing arg2"),
}
```
Use `match` on the `Option` returned from `next`

### Step 3 - Specify lifetime of return
`'static` why?
Because now we are for sure only return a static string.
Previously lifetime was inferred from the borrowing, both would have the same lifetime and was inferred. Now Args is taken ownership of which makes it not be able to infer lifetime from a borrowed value.
```rust
pub fn new(mut args: env::Args) -> Result<Config, &'static str>
```

### Step 4 - Making things clearer with `iterator adaptors`
In the search function, we had a `for` to iterate over the values.
Change it so that it is easier to understand
```rust
pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    contents
        .lines()
        .filter(|x| x.contains(query))
        .map(|x| x.trim())
        .collect()
}
```

===

# Comparing Performance
In theory, Iterators are one of the `zero-cost abstractions`, basically, it is done in a way that it does not affect performance over base way of doing things.