mod Aes;


pub use Aes::Aes128Cipher;
pub use Aes::Aes192Cipher;
pub use Aes::Aes256Cipher;


pub trait BlockCipherTrait
{
	const BLOCK_SIZE: usize;

	fn EncryptBlock (&self, input: &[u8; Self::BLOCK_SIZE]) -> [u8; Self::BLOCK_SIZE];
	fn DecryptBlock (&self, input: &[u8; Self::BLOCK_SIZE]) -> [u8; Self::BLOCK_SIZE];

	fn EncryptBlockInplace (&self, input: &[u8; Self::BLOCK_SIZE], output: &mut [u8; Self::BLOCK_SIZE]);
	fn DecryptBlockInplace (&self, input: &[u8; Self::BLOCK_SIZE], output: &mut [u8; Self::BLOCK_SIZE]);
}


#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum BlockMode
{
	ECB,
	CBC,
	OFB,
	CFB,
	CTR,
	XTS,
}


#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum PaddingMode
{
	PKCS,
}


#[derive(Debug, Clone)]
pub struct BlockCipher<T: BlockCipherTrait>
where [u8; T::BLOCK_SIZE]:
{
	cipher: T,
	blockMode: BlockMode,
	paddingMode: PaddingMode,
	iv: [u8; T::BLOCK_SIZE],
}

impl<T: BlockCipherTrait> BlockCipher<T>
where [u8; T::BLOCK_SIZE]:
{
	pub fn New (cipher: T, blockMode: BlockMode, paddingMode: PaddingMode, iv: &[u8]) -> Self
	{
		Self
		{
			cipher,
			blockMode,
			paddingMode,
			iv: iv.try_into ().unwrap (),
		}
	}

	fn XorBlock (a: &[u8; T::BLOCK_SIZE], b: &[u8; T::BLOCK_SIZE]) -> [u8; T::BLOCK_SIZE]
	{
		let mut output = [0; T::BLOCK_SIZE];

		for i in 0..output.len ()
		{
			output[i] = a[i] ^ b[i];
		}

		output
	}

	fn XorBlockInplace (a: &[u8; T::BLOCK_SIZE], b: &mut [u8; T::BLOCK_SIZE])
	{
		for i in 0..T::BLOCK_SIZE
		{
			b[i] ^= a[i];
		}
	}

	// include padding
	fn MakeLastBlock (&self, input: &[u8]) -> [u8; T::BLOCK_SIZE]
	{
		let mut output = [0; T::BLOCK_SIZE];

		match self.paddingMode
		{
			PaddingMode::PKCS =>
			{
				// calc padding size
				let leftOverSize = ((input.len () + T::BLOCK_SIZE - 1)/T::BLOCK_SIZE)*T::BLOCK_SIZE - input.len ();
				let paddingSize = T::BLOCK_SIZE - leftOverSize;

				// first part
				for i in 0..leftOverSize
				{
					output[i] = input[input.len () - leftOverSize + i];
				}

				for i in leftOverSize..T::BLOCK_SIZE
				{
					output[i] = paddingSize.try_into ().unwrap ();
				}
			}
		}

		output
	}

	// Assume output is big enough encrypted input, included padding
	fn CBCEncryptInplace (&self, input: &[u8], output: &mut [u8])
	{
		// calc output size
		let expectedOutputSize = if input.len () % T::BLOCK_SIZE == 0
		{
			input.len () + T::BLOCK_SIZE
		}
		else
		{
			((input.len () + T::BLOCK_SIZE - 1)/T::BLOCK_SIZE)*T::BLOCK_SIZE
		};

		// check output size is correct
		assert! (output.len () >= expectedOutputSize);
		let lastBlock = self.MakeLastBlock (input);

		let mut currentXorBlock = self.iv;

		for (i, inputBlock) in input.chunks_exact (T::BLOCK_SIZE).enumerate ()
		{
			let i = i*T::BLOCK_SIZE;

			Self::XorBlockInplace (inputBlock.try_into ().unwrap (), &mut currentXorBlock);
			self.cipher.EncryptBlockInplace (&currentXorBlock, (&mut output[i..i + T::BLOCK_SIZE]).try_into ().unwrap ());

			currentXorBlock = output[i..i + T::BLOCK_SIZE].try_into ().unwrap ();
		}

		// last chunk
		let outputLen = output.len ();
		Self::XorBlockInplace (&lastBlock, &mut currentXorBlock);
		self.cipher.EncryptBlockInplace (&currentXorBlock, (&mut 
			output[outputLen - T::BLOCK_SIZE..outputLen]).try_into ().unwrap ());
	}

	// Assume output is big enough encrypted input, included padding
	fn CBCDecryptInplace (&self, input: &[u8], output: &mut [u8]) -> usize
	{
		assert! (input.len () % T::BLOCK_SIZE == 0);
		assert! (output.len () >= input.len ());

		let mut currentXorBlock = self.iv;

		for (i, inputBlock) in input.chunks_exact (T::BLOCK_SIZE).enumerate ()
		{
			let i = i*T::BLOCK_SIZE;

			self.cipher.DecryptBlockInplace (inputBlock.try_into ().unwrap (), (&mut output[i..i + T::BLOCK_SIZE]).try_into ().unwrap ());
			Self::XorBlockInplace (&currentXorBlock, (&mut output[i..i + T::BLOCK_SIZE]).try_into ().unwrap ());

			currentXorBlock = inputBlock.try_into ().unwrap ();
		}

		// check the last byte according to padding rule
		match self.paddingMode
		{
			PaddingMode::PKCS => input.len () - output[input.len () - 1] as usize,
		}
	}

	/// output must big enough to hold encrypted input, included padding
	pub fn EncryptInplace (&self, input: &[u8], output: &mut [u8])
	{
		assert! (output.len () >= input.len ());

		match self.blockMode
		{
			BlockMode::CBC => self.CBCEncryptInplace (input, output),
			_ => panic! ("unimplemented!"),
		}
	}

	/// return plaintext size accroding to padding rule
	/// output must big enough to hold decrypted input
	pub fn DecryptInplace (&self, input: &[u8], output: &mut [u8]) -> usize
	{
		assert! (input.len () % T::BLOCK_SIZE == 0);
		assert! (output.len () >= input.len ());

		match self.blockMode
		{
			BlockMode::CBC => self.CBCDecryptInplace (input, output),
			_ => panic! ("unimplemented!"),
		}
	}

	pub fn Encrypt (&self, input: &[u8]) -> Vec<u8>
	{
		// calc output size
		let expectedOutputSize = if input.len () % T::BLOCK_SIZE == 0
		{
			input.len () + T::BLOCK_SIZE
		}
		else
		{
			((input.len () + T::BLOCK_SIZE - 1)/T::BLOCK_SIZE)*T::BLOCK_SIZE
		};

		let mut output = vec! [0; expectedOutputSize];
		self.EncryptInplace (input, &mut output);

		output
	}

	// decrypt the input, trip the output according to padding mode or not
	pub fn Decrypt (&self, input: &[u8], stripOutput: bool) -> Vec<u8>
	{
		// just make a big enough buffer first
		// calc output size
		let expectedOutputSize = if input.len () % T::BLOCK_SIZE == 0
		{
			input.len () + T::BLOCK_SIZE
		}
		else
		{
			((input.len () + T::BLOCK_SIZE - 1)/T::BLOCK_SIZE)*T::BLOCK_SIZE
		};

		let mut output = vec! [0; expectedOutputSize];
		let plainTextLen = self.DecryptInplace (input, &mut output);

		if stripOutput
		{
			output.resize (plainTextLen, 0);
		}

		output
	}
}

