use crate::BlockCipher::BlockCipher;


static RoundConstant: [u8; 11] = [ 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36 ];

static SBox: [u8; 256] =
[
	0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76,
	0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0,
	0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15,
	0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75,
	0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84,
	0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf,
	0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8,
	0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2,
	0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73,
	0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb,
	0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79,
	0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08,
	0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a,
	0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e,
	0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf,
	0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16
];

static InvertSBox: [u8; 256] =
[
	0x52, 0x09, 0x6a, 0xd5, 0x30, 0x36, 0xa5, 0x38, 0xbf, 0x40, 0xa3, 0x9e, 0x81, 0xf3, 0xd7, 0xfb,
	0x7c, 0xe3, 0x39, 0x82, 0x9b, 0x2f, 0xff, 0x87, 0x34, 0x8e, 0x43, 0x44, 0xc4, 0xde, 0xe9, 0xcb,
	0x54, 0x7b, 0x94, 0x32, 0xa6, 0xc2, 0x23, 0x3d, 0xee, 0x4c, 0x95, 0x0b, 0x42, 0xfa, 0xc3, 0x4e,
	0x08, 0x2e, 0xa1, 0x66, 0x28, 0xd9, 0x24, 0xb2, 0x76, 0x5b, 0xa2, 0x49, 0x6d, 0x8b, 0xd1, 0x25,
	0x72, 0xf8, 0xf6, 0x64, 0x86, 0x68, 0x98, 0x16, 0xd4, 0xa4, 0x5c, 0xcc, 0x5d, 0x65, 0xb6, 0x92,
	0x6c, 0x70, 0x48, 0x50, 0xfd, 0xed, 0xb9, 0xda, 0x5e, 0x15, 0x46, 0x57, 0xa7, 0x8d, 0x9d, 0x84,
	0x90, 0xd8, 0xab, 0x00, 0x8c, 0xbc, 0xd3, 0x0a, 0xf7, 0xe4, 0x58, 0x05, 0xb8, 0xb3, 0x45, 0x06,
	0xd0, 0x2c, 0x1e, 0x8f, 0xca, 0x3f, 0x0f, 0x02, 0xc1, 0xaf, 0xbd, 0x03, 0x01, 0x13, 0x8a, 0x6b,
	0x3a, 0x91, 0x11, 0x41, 0x4f, 0x67, 0xdc, 0xea, 0x97, 0xf2, 0xcf, 0xce, 0xf0, 0xb4, 0xe6, 0x73,
	0x96, 0xac, 0x74, 0x22, 0xe7, 0xad, 0x35, 0x85, 0xe2, 0xf9, 0x37, 0xe8, 0x1c, 0x75, 0xdf, 0x6e,
	0x47, 0xf1, 0x1a, 0x71, 0x1d, 0x29, 0xc5, 0x89, 0x6f, 0xb7, 0x62, 0x0e, 0xaa, 0x18, 0xbe, 0x1b,
	0xfc, 0x56, 0x3e, 0x4b, 0xc6, 0xd2, 0x79, 0x20, 0x9a, 0xdb, 0xc0, 0xfe, 0x78, 0xcd, 0x5a, 0xf4,
	0x1f, 0xdd, 0xa8, 0x33, 0x88, 0x07, 0xc7, 0x31, 0xb1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xec, 0x5f,
	0x60, 0x51, 0x7f, 0xa9, 0x19, 0xb5, 0x4a, 0x0d, 0x2d, 0xe5, 0x7a, 0x9f, 0x93, 0xc9, 0x9c, 0xef,
	0xa0, 0xe0, 0x3b, 0x4d, 0xae, 0x2a, 0xf5, 0xb0, 0xc8, 0xeb, 0xbb, 0x3c, 0x83, 0x53, 0x99, 0x61,
	0x17, 0x2b, 0x04, 0x7e, 0xba, 0x77, 0xd6, 0x26, 0xe1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0c, 0x7d
];


fn AddRoundKey (state: &mut [[u8; 4]; 4], roundKey: &[[u8; 4]; 4])
{
	for i in 0..4
	{
		for j in 0..4
		{
			state[i][j] = state[i][j] ^ roundKey[i][j];
		}
	}
}

fn SubBytes (state: &mut [[u8; 4]; 4])
{
	for i in 0..4
	{
		for j in 0..4
		{
			state[i][j] = SBox[state[i][j] as usize];
		}
	}
}

fn ShiftRows (state: &mut [[u8; 4]; 4])
{
	let tmp = state[1][0];
	state[1][0] = state[1][1];
	state[1][1] = state[1][2];
	state[1][2] = state[1][3];
	state[1][3] = tmp;

	let tmp = state[2][0];
	state[2][0] = state[2][2];
	state[2][2] = tmp;

	let tmp = state[2][1];
	state[2][1] = state[2][3];
	state[2][3] = tmp;

	let tmp = state[3][3];
	state[3][3] = state[3][2];
	state[3][2] = state[3][1];
	state[3][1] = state[3][0];
	state[3][0] = tmp;
}

fn MixColumns (state: &mut [[u8; 4]; 4])
{
	const multiplicateMatix: [[u8; 4]; 4] =
	[
		[2, 3, 1, 1],
		[1, 2, 3, 1],
		[1, 1, 2, 3],
		[3, 1, 1, 2],
	];

	// matrix multiplication
	// [2 3 1 1]*[a0j] = [b0j]
	// [1 2 3 1]*[a1j] = [b1j]
	// [1 1 2 3]*[a2j] = [b2j]
	// [3 1 1 2]*[a3j] = [b3j]


	for j in 0..4
	{
		let mut tmp = [0; 4];

		for i in 0..4
		{
			tmp[i] = multiplicateMatix[i][0] * state[0][j]
				+ multiplicateMatix[i][1] * state[1][j]
				+ multiplicateMatix[i][2] * state[2][j]
				+ multiplicateMatix[i][3] * state[3][j];
		}

		for i in 0..4
		{
			state[i][j] = tmp[i];
		}
	}
}

fn InvertSubBytes (state: &mut [[u8; 4]; 4])
{
	for i in 0..4
	{
		for j in 0..4
		{
			state[i][j] = InvertSBox[state[i][j] as usize];
		}
	}
}

fn InvertShiftRows (state: &mut [[u8; 4]; 4])
{
	let tmp = state[1][3];
	state[1][3] = state[1][2];
	state[1][2] = state[1][1];
	state[1][1] = state[1][0];
	state[1][0] = tmp;

	let tmp = state[2][2];
	state[2][2] = state[2][0];
	state[2][0] = tmp;

	let tmp = state[2][3];
	state[2][3] = state[2][1];
	state[2][1] = tmp;

	let tmp = state[3][0];
	state[3][0] = state[3][1];
	state[3][1] = state[3][2];
	state[3][2] = state[3][3];
	state[3][3] = tmp;
}

fn InvertMixColumns (state: &mut [[u8; 4]; 4])
{
	const multiplicateMatix: [[u8; 4]; 4] =
	[
		[14, 11, 13, 9],
		[9, 14, 11, 13],
		[13, 9, 14, 11],
		[11, 13, 9, 14],
	];

	// matrix multiplication
	// [14 11 13 9]*[a0j] = [b0j]
	// [9 14 11 13]*[a1j] = [b1j]
	// [13 9 14 11]*[a2j] = [b2j]
	// [11 13 9 14]*[a3j] = [b3j]


	for j in 0..4
	{
		let mut tmp = [0; 4];

		for i in 0..4
		{
			tmp[i] = multiplicateMatix[i][0] * state[0][j]
				+ multiplicateMatix[i][1] * state[1][j]
				+ multiplicateMatix[i][2] * state[2][j]
				+ multiplicateMatix[i][3] * state[3][j];
		}

		for i in 0..4
		{
			state[i][j] = tmp[i];
		}
	}
}


#[derive(Debug, Clone)]
pub struct Aes128Cipher
{
	key: [u8; Self::KEY_WORD_LEN*4],
	roundKeys: [[[u8; 4]; 4]; Self::ROUND_NUMBER + 1],
}

impl Aes128Cipher
{
	const KEY_WORD_LEN: usize = 4;
	const ROUND_NUMBER: usize = 10;
	const EXPANDED_KEY_WORD_LEN: usize = 4*(Self::ROUND_NUMBER + 1);


	pub fn New (key: &[u8; Self::KEY_WORD_LEN*4]) -> Self
	{
		let mut result = Self
		{
			key: key.clone (),
			roundKeys: [[[0u8; 4]; 4]; Self::ROUND_NUMBER + 1],
		};

		result.CreateRoundKeys ();
		result
	}

	fn CreateRoundKeys (&mut self)
	{
		let mut expandedKey = [[0; 4]; Self::EXPANDED_KEY_WORD_LEN];

		for i in 0..Self::EXPANDED_KEY_WORD_LEN
		{
			if i < Self::KEY_WORD_LEN
			{
				expandedKey[i][0] = self.key[i*4 + 0];
				expandedKey[i][1] = self.key[i*4 + 1];
				expandedKey[i][2] = self.key[i*4 + 2];
				expandedKey[i][3] = self.key[i*4 + 3];
			}
			else if i % Self::KEY_WORD_LEN == 0
			{
				// get last key
				let mut tmpWord = expandedKey[i - 1];

				// shift to the left
				let tmp = tmpWord[0];
				tmpWord[0] = tmpWord[1];
				tmpWord[1] = tmpWord[2];
				tmpWord[2] = tmpWord[3];
				tmpWord[3] = tmp;

				// sub each word
				tmpWord[0] = SBox[tmpWord[0] as usize];
				tmpWord[1] = SBox[tmpWord[1] as usize];
				tmpWord[2] = SBox[tmpWord[2] as usize];
				tmpWord[3] = SBox[tmpWord[3] as usize];

				// do stuff with round constant
				tmpWord[0] = tmpWord[0] ^ RoundConstant[i/Self::KEY_WORD_LEN];

				expandedKey[i][0] = expandedKey[i - Self::KEY_WORD_LEN][0] ^ tmpWord[0];
				expandedKey[i][1] = expandedKey[i - Self::KEY_WORD_LEN][1] ^ tmpWord[1];
				expandedKey[i][2] = expandedKey[i - Self::KEY_WORD_LEN][2] ^ tmpWord[2];
				expandedKey[i][3] = expandedKey[i - Self::KEY_WORD_LEN][3] ^ tmpWord[3];
			}
			else if Self::KEY_WORD_LEN > 6 && (i % Self::KEY_WORD_LEN) == 4
			{
				// get last key
				let mut tmpWord = expandedKey[i - 1];

				// sub each word
				tmpWord[0] = SBox[tmpWord[0] as usize];
				tmpWord[1] = SBox[tmpWord[1] as usize];
				tmpWord[2] = SBox[tmpWord[2] as usize];
				tmpWord[3] = SBox[tmpWord[3] as usize];

				expandedKey[i][0] = expandedKey[i - Self::KEY_WORD_LEN][0] ^ tmpWord[0];
				expandedKey[i][1] = expandedKey[i - Self::KEY_WORD_LEN][1] ^ tmpWord[1];
				expandedKey[i][2] = expandedKey[i - Self::KEY_WORD_LEN][2] ^ tmpWord[2];
				expandedKey[i][3] = expandedKey[i - Self::KEY_WORD_LEN][3] ^ tmpWord[3];
			}
			else
			{
				expandedKey[i][0] = expandedKey[i - Self::KEY_WORD_LEN][0] ^ expandedKey[i - 1][0];
				expandedKey[i][1] = expandedKey[i - Self::KEY_WORD_LEN][1] ^ expandedKey[i - 1][1];
				expandedKey[i][2] = expandedKey[i - Self::KEY_WORD_LEN][2] ^ expandedKey[i - 1][2];
				expandedKey[i][3] = expandedKey[i - Self::KEY_WORD_LEN][3] ^ expandedKey[i - 1][3];
			}
		}

		for i in 0..Self::ROUND_NUMBER + 1
		{
			for j in 0..4
			{
				self.roundKeys[i][0][j] = expandedKey[i*4 + j][0];
				self.roundKeys[i][1][j] = expandedKey[i*4 + j][1];
				self.roundKeys[i][2][j] = expandedKey[i*4 + j][2];
				self.roundKeys[i][3][j] = expandedKey[i*4 + j][3];
			}
		}
	}

	pub fn Encrypt (&self, input: &[u8; 16]) -> [u8; 16]
	{
		// transmute input to state array
		let mut state = [[0; 4]; 4];

		for i in 0..4
		{
			for j in 0..4
			{
				state[j][i] = input[i*4 + j];
			}
		}

		// init round key addition
		AddRoundKey (&mut state, &self.roundKeys[0]);

		for i in 1..Self::ROUND_NUMBER
		{
			SubBytes (&mut state);
			ShiftRows (&mut state);
			MixColumns (&mut state);
			AddRoundKey (&mut state, &self.roundKeys[i]);
		}

		// final round
		SubBytes (&mut state);
		ShiftRows (&mut state);
		AddRoundKey (&mut state, &self.roundKeys[Self::ROUND_NUMBER]);


		// transmute state array into output
		let mut output = [0; 16];

		for i in 0..4
		{
			for j in 0..4
			{
				output[i*4 + j] = state[j][i];
			}
		}

		output
	}

	pub fn Decrypt (&self, input: &[u8; 16]) -> [u8; 16]
	{
		// transmute input to state array
		let mut state = [[0; 4]; 4];

		for i in 0..4
		{
			for j in 0..4
			{
				state[j][i] = input[i*4 + j];
			}
		}

		// reverse last round
		AddRoundKey (&mut state, &self.roundKeys[Self::ROUND_NUMBER]);
		InvertShiftRows (&mut state);
		InvertSubBytes (&mut state);

		for i in (1..Self::ROUND_NUMBER).rev ()
		{
			AddRoundKey (&mut state, &self.roundKeys[i]);
			InvertMixColumns (&mut state);
			InvertShiftRows (&mut state);
			InvertSubBytes (&mut state);
		}

		// reverse init round key addition
		AddRoundKey (&mut state, &self.roundKeys[0]);


		// transmute state array into output
		let mut output = [0; 16];

		for i in 0..4
		{
			for j in 0..4
			{
				output[i*4 + j] = state[j][i];
			}
		}

		output
	}
}


impl BlockCipher for Aes128Cipher
{
	fn BlockSize (&self) -> usize { 16 }

	fn EncryptBlock (&self, input: &[u8]) -> Vec<u8>
	{
		assert!(input.len () == self.BlockSize ());
		self.Encrypt (input.try_into ().unwrap ()).to_vec ()
	}

	fn DecryptBlock (&self, input: &[u8]) -> Vec<u8>
	{
		assert!(input.len () == self.BlockSize ());
		self.Decrypt (input.try_into ().unwrap ()).to_vec ()
	}
}


#[derive(Debug, Clone)]
pub struct Aes192Cipher
{
	key: [u8; Self::KEY_WORD_LEN*4],
	roundKeys: [[[u8; 4]; 4]; Self::ROUND_NUMBER + 1],
}

impl Aes192Cipher
{
	const KEY_WORD_LEN: usize = 6;
	const ROUND_NUMBER: usize = 12;
	const EXPANDED_KEY_WORD_LEN: usize = 4*(Self::ROUND_NUMBER + 1);


	pub fn New (key: &[u8; Self::KEY_WORD_LEN*4]) -> Self
	{
		let mut result = Self
		{
			key: key.clone (),
			roundKeys: [[[0u8; 4]; 4]; Self::ROUND_NUMBER + 1],
		};

		result.CreateRoundKeys ();
		result
	}

	fn CreateRoundKeys (&mut self)
	{
		let mut expandedKey = [[0; 4]; Self::EXPANDED_KEY_WORD_LEN];

		for i in 0..Self::EXPANDED_KEY_WORD_LEN
		{
			if i < Self::KEY_WORD_LEN
			{
				expandedKey[i][0] = self.key[i*4 + 0];
				expandedKey[i][1] = self.key[i*4 + 1];
				expandedKey[i][2] = self.key[i*4 + 2];
				expandedKey[i][3] = self.key[i*4 + 3];
			}
			else if i % Self::KEY_WORD_LEN == 0
			{
				// get last key
				let mut tmpWord = expandedKey[i - 1];

				// shift to the left
				let tmp = tmpWord[0];
				tmpWord[0] = tmpWord[1];
				tmpWord[1] = tmpWord[2];
				tmpWord[2] = tmpWord[3];
				tmpWord[3] = tmp;

				// sub each word
				tmpWord[0] = SBox[tmpWord[0] as usize];
				tmpWord[1] = SBox[tmpWord[1] as usize];
				tmpWord[2] = SBox[tmpWord[2] as usize];
				tmpWord[3] = SBox[tmpWord[3] as usize];

				// do stuff with round constant
				tmpWord[0] = tmpWord[0] ^ RoundConstant[i/Self::KEY_WORD_LEN];

				expandedKey[i][0] = expandedKey[i - Self::KEY_WORD_LEN][0] ^ tmpWord[0];
				expandedKey[i][1] = expandedKey[i - Self::KEY_WORD_LEN][1] ^ tmpWord[1];
				expandedKey[i][2] = expandedKey[i - Self::KEY_WORD_LEN][2] ^ tmpWord[2];
				expandedKey[i][3] = expandedKey[i - Self::KEY_WORD_LEN][3] ^ tmpWord[3];
			}
			else if Self::KEY_WORD_LEN > 6 && (i % Self::KEY_WORD_LEN) == 4
			{
				// get last key
				let mut tmpWord = expandedKey[i - 1];

				// sub each word
				tmpWord[0] = SBox[tmpWord[0] as usize];
				tmpWord[1] = SBox[tmpWord[1] as usize];
				tmpWord[2] = SBox[tmpWord[2] as usize];
				tmpWord[3] = SBox[tmpWord[3] as usize];

				expandedKey[i][0] = expandedKey[i - Self::KEY_WORD_LEN][0] ^ tmpWord[0];
				expandedKey[i][1] = expandedKey[i - Self::KEY_WORD_LEN][1] ^ tmpWord[1];
				expandedKey[i][2] = expandedKey[i - Self::KEY_WORD_LEN][2] ^ tmpWord[2];
				expandedKey[i][3] = expandedKey[i - Self::KEY_WORD_LEN][3] ^ tmpWord[3];
			}
			else
			{
				expandedKey[i][0] = expandedKey[i - Self::KEY_WORD_LEN][0] ^ expandedKey[i - 1][0];
				expandedKey[i][1] = expandedKey[i - Self::KEY_WORD_LEN][1] ^ expandedKey[i - 1][1];
				expandedKey[i][2] = expandedKey[i - Self::KEY_WORD_LEN][2] ^ expandedKey[i - 1][2];
				expandedKey[i][3] = expandedKey[i - Self::KEY_WORD_LEN][3] ^ expandedKey[i - 1][3];
			}
		}

		for i in 0..Self::ROUND_NUMBER + 1
		{
			for j in 0..4
			{
				self.roundKeys[i][0][j] = expandedKey[i*4 + j][0];
				self.roundKeys[i][1][j] = expandedKey[i*4 + j][1];
				self.roundKeys[i][2][j] = expandedKey[i*4 + j][2];
				self.roundKeys[i][3][j] = expandedKey[i*4 + j][3];
			}
		}
	}

	pub fn Encrypt (&self, input: &[u8; 16]) -> [u8; 16]
	{
		// transmute input to state array
		let mut state = [[0; 4]; 4];

		for i in 0..4
		{
			for j in 0..4
			{
				state[j][i] = input[i*4 + j];
			}
		}

		// init round key addition
		AddRoundKey (&mut state, &self.roundKeys[0]);

		for i in 1..Self::ROUND_NUMBER
		{
			SubBytes (&mut state);
			ShiftRows (&mut state);
			MixColumns (&mut state);
			AddRoundKey (&mut state, &self.roundKeys[i]);
		}

		// final round
		SubBytes (&mut state);
		ShiftRows (&mut state);
		AddRoundKey (&mut state, &self.roundKeys[Self::ROUND_NUMBER]);


		// transmute state array into output
		let mut output = [0; 16];

		for i in 0..4
		{
			for j in 0..4
			{
				output[i*4 + j] = state[j][i];
			}
		}

		output
	}

	pub fn Decrypt (&self, input: &[u8; 16]) -> [u8; 16]
	{
		// transmute input to state array
		let mut state = [[0; 4]; 4];

		for i in 0..4
		{
			for j in 0..4
			{
				state[j][i] = input[i*4 + j];
			}
		}

		// reverse last round
		AddRoundKey (&mut state, &self.roundKeys[Self::ROUND_NUMBER]);
		InvertShiftRows (&mut state);
		InvertSubBytes (&mut state);

		for i in (1..Self::ROUND_NUMBER).rev ()
		{
			AddRoundKey (&mut state, &self.roundKeys[i]);
			InvertMixColumns (&mut state);
			InvertShiftRows (&mut state);
			InvertSubBytes (&mut state);
		}

		// reverse init round key addition
		AddRoundKey (&mut state, &self.roundKeys[0]);


		// transmute state array into output
		let mut output = [0; 16];

		for i in 0..4
		{
			for j in 0..4
			{
				output[i*4 + j] = state[j][i];
			}
		}

		output
	}
}


impl BlockCipher for Aes192Cipher
{
	fn BlockSize (&self) -> usize { 16 }

	fn EncryptBlock (&self, input: &[u8]) -> Vec<u8>
	{
		assert!(input.len () == self.BlockSize ());
		self.Encrypt (input.try_into ().unwrap ()).to_vec ()
	}

	fn DecryptBlock (&self, input: &[u8]) -> Vec<u8>
	{
		assert!(input.len () == self.BlockSize ());
		self.Decrypt (input.try_into ().unwrap ()).to_vec ()
	}
}


#[derive(Debug, Clone)]
pub struct Aes256Cipher
{
	key: [u8; Self::KEY_WORD_LEN*4],
	roundKeys: [[[u8; 4]; 4]; Self::ROUND_NUMBER + 1],
}

impl Aes256Cipher
{
	const KEY_WORD_LEN: usize = 8;
	const ROUND_NUMBER: usize = 14;
	const EXPANDED_KEY_WORD_LEN: usize = 4*(Self::ROUND_NUMBER + 1);


	pub fn New (key: &[u8; Self::KEY_WORD_LEN*4]) -> Self
	{
		let mut result = Self
		{
			key: key.clone (),
			roundKeys: [[[0u8; 4]; 4]; Self::ROUND_NUMBER + 1],
		};

		result.CreateRoundKeys ();
		result
	}

	fn CreateRoundKeys (&mut self)
	{
		let mut expandedKey = [[0; 4]; Self::EXPANDED_KEY_WORD_LEN];

		for i in 0..Self::EXPANDED_KEY_WORD_LEN
		{
			if i < Self::KEY_WORD_LEN
			{
				expandedKey[i][0] = self.key[i*4 + 0];
				expandedKey[i][1] = self.key[i*4 + 1];
				expandedKey[i][2] = self.key[i*4 + 2];
				expandedKey[i][3] = self.key[i*4 + 3];
			}
			else if i % Self::KEY_WORD_LEN == 0
			{
				// get last key
				let mut tmpWord = expandedKey[i - 1];

				// shift to the left
				let tmp = tmpWord[0];
				tmpWord[0] = tmpWord[1];
				tmpWord[1] = tmpWord[2];
				tmpWord[2] = tmpWord[3];
				tmpWord[3] = tmp;

				// sub each word
				tmpWord[0] = SBox[tmpWord[0] as usize];
				tmpWord[1] = SBox[tmpWord[1] as usize];
				tmpWord[2] = SBox[tmpWord[2] as usize];
				tmpWord[3] = SBox[tmpWord[3] as usize];

				// do stuff with round constant
				tmpWord[0] = tmpWord[0] ^ RoundConstant[i/Self::KEY_WORD_LEN];

				expandedKey[i][0] = expandedKey[i - Self::KEY_WORD_LEN][0] ^ tmpWord[0];
				expandedKey[i][1] = expandedKey[i - Self::KEY_WORD_LEN][1] ^ tmpWord[1];
				expandedKey[i][2] = expandedKey[i - Self::KEY_WORD_LEN][2] ^ tmpWord[2];
				expandedKey[i][3] = expandedKey[i - Self::KEY_WORD_LEN][3] ^ tmpWord[3];
			}
			else if Self::KEY_WORD_LEN > 6 && (i % Self::KEY_WORD_LEN) == 4
			{
				// get last key
				let mut tmpWord = expandedKey[i - 1];

				// sub each word
				tmpWord[0] = SBox[tmpWord[0] as usize];
				tmpWord[1] = SBox[tmpWord[1] as usize];
				tmpWord[2] = SBox[tmpWord[2] as usize];
				tmpWord[3] = SBox[tmpWord[3] as usize];

				expandedKey[i][0] = expandedKey[i - Self::KEY_WORD_LEN][0] ^ tmpWord[0];
				expandedKey[i][1] = expandedKey[i - Self::KEY_WORD_LEN][1] ^ tmpWord[1];
				expandedKey[i][2] = expandedKey[i - Self::KEY_WORD_LEN][2] ^ tmpWord[2];
				expandedKey[i][3] = expandedKey[i - Self::KEY_WORD_LEN][3] ^ tmpWord[3];
			}
			else
			{
				expandedKey[i][0] = expandedKey[i - Self::KEY_WORD_LEN][0] ^ expandedKey[i - 1][0];
				expandedKey[i][1] = expandedKey[i - Self::KEY_WORD_LEN][1] ^ expandedKey[i - 1][1];
				expandedKey[i][2] = expandedKey[i - Self::KEY_WORD_LEN][2] ^ expandedKey[i - 1][2];
				expandedKey[i][3] = expandedKey[i - Self::KEY_WORD_LEN][3] ^ expandedKey[i - 1][3];
			}
		}

		for i in 0..Self::ROUND_NUMBER + 1
		{
			for j in 0..4
			{
				self.roundKeys[i][0][j] = expandedKey[i*4 + j][0];
				self.roundKeys[i][1][j] = expandedKey[i*4 + j][1];
				self.roundKeys[i][2][j] = expandedKey[i*4 + j][2];
				self.roundKeys[i][3][j] = expandedKey[i*4 + j][3];
			}
		}
	}

	pub fn Encrypt (&self, input: &[u8; 16]) -> [u8; 16]
	{
		// transmute input to state array
		let mut state = [[0; 4]; 4];

		for i in 0..4
		{
			for j in 0..4
			{
				state[j][i] = input[i*4 + j];
			}
		}

		// init round key addition
		AddRoundKey (&mut state, &self.roundKeys[0]);

		for i in 1..Self::ROUND_NUMBER
		{
			SubBytes (&mut state);
			ShiftRows (&mut state);
			MixColumns (&mut state);
			AddRoundKey (&mut state, &self.roundKeys[i]);
		}

		// final round
		SubBytes (&mut state);
		ShiftRows (&mut state);
		AddRoundKey (&mut state, &self.roundKeys[Self::ROUND_NUMBER]);


		// transmute state array into output
		let mut output = [0; 16];

		for i in 0..4
		{
			for j in 0..4
			{
				output[i*4 + j] = state[j][i];
			}
		}

		output
	}

	pub fn Decrypt (&self, input: &[u8; 16]) -> [u8; 16]
	{
		// transmute input to state array
		let mut state = [[0; 4]; 4];

		for i in 0..4
		{
			for j in 0..4
			{
				state[j][i] = input[i*4 + j];
			}
		}

		// reverse last round
		AddRoundKey (&mut state, &self.roundKeys[Self::ROUND_NUMBER]);
		InvertShiftRows (&mut state);
		InvertSubBytes (&mut state);

		for i in (1..Self::ROUND_NUMBER).rev ()
		{
			AddRoundKey (&mut state, &self.roundKeys[i]);
			InvertMixColumns (&mut state);
			InvertShiftRows (&mut state);
			InvertSubBytes (&mut state);
		}

		// reverse init round key addition
		AddRoundKey (&mut state, &self.roundKeys[0]);


		// transmute state array into output
		let mut output = [0; 16];

		for i in 0..4
		{
			for j in 0..4
			{
				output[i*4 + j] = state[j][i];
			}
		}

		output
	}
}


impl BlockCipher for Aes256Cipher
{
	fn BlockSize (&self) -> usize { 16 }

	fn EncryptBlock (&self, input: &[u8]) -> Vec<u8>
	{
		assert!(input.len () == self.BlockSize ());
		self.Encrypt (input.try_into ().unwrap ()).to_vec ()
	}

	fn DecryptBlock (&self, input: &[u8]) -> Vec<u8>
	{
		assert!(input.len () == self.BlockSize ());
		self.Decrypt (input.try_into ().unwrap ()).to_vec ()
	}
}
