use regex::Regex;

/// Lists validation for multiple cryptocurrency addresses
pub struct CanaryValidateCryptocurrencyAddressAPI;
pub struct CanaryValidateEncodingAPI;
pub struct CanaryValidateOtherAPI;
pub struct CanaryValidateMnemonicPhraseAPI;
pub struct CanaryValidateDigitAPI;
pub struct CanaryValidateBooleanAPI;
pub struct CanaryValidateIPAddressAPI;
pub struct CanaryValidateDateAndTimeAPI;
pub struct CanaryValidateDomainName;

pub struct CanaryStandard;

impl CanaryStandard {
    pub fn phone_number(){
        //
    }
}

impl CanaryValidateDomainName {
    pub fn domain_name<T: AsRef<str>>(input: T) -> bool {
        let re = Regex::new(r"^((?!-)[A-Za-z0-9-]{1, 63}(?<!-)\\.)+[A-Za-z]{2, 6}$").unwrap();
        return re.is_match(input.as_ref())
    }
}

impl CanaryValidateDateAndTimeAPI {
    pub fn date<T: AsRef<str>>(input: T) -> bool {
        let re = Regex::new(r"^(19|20)\d\d[- /.](0[1-9]|1[012])[- /.](0[1-9]|[12][0-9]|3[01])$").unwrap();
        return re.is_match(input.as_ref())
    }
    pub fn utc_timestamp<T: AsRef<str>>(input: T) -> bool {
        let re = Regex::new(r"^\\d{4}-[0-1][0-3]-[0-3]\\d{1}T[0-2]\\d{1}:[0-5]\\d{1}:[0-5]\\d{1}Z$").unwrap();
        return re.is_match(input.as_ref())
    }
}

impl CanaryValidateIPAddressAPI {
    pub fn ip_address<T: AsRef<str>>(input: T) -> bool {
        let re = Regex::new(r"\b\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}\b").unwrap();
        return re.is_match(input.as_ref())
    }
}

impl CanaryValidateBooleanAPI {
    pub fn true_or_false<T: AsRef<str>>(input: T) -> bool {
        let re = Regex::new(r"^(?i)(true|false)$").unwrap();
        return re.is_match(input.as_ref())
    }
    pub fn zero_or_one_str<T: AsRef<str>>(input: T) -> bool {
        if input.as_ref() == "0" || input.as_ref() == "1" {
            return true
        }
        else {
            return false
        }
    }
}

/// # Digit
/// 
/// Digit validates numbers
impl CanaryValidateDigitAPI {
    pub fn single_digit<T: AsRef<str>>(digit: T) -> bool {
        let re = Regex::new(r"^\d$").unwrap();
        return re.is_match(digit.as_ref())
    }
}

impl CanaryValidateOtherAPI {
    pub fn email_address<T: AsRef<str>>(email: T) -> bool{
        let re = Regex::new(r"^([\w\-\.]+)@((\[([0-9]{1,3}\.){3}[0-9]{1,3}\])|(([\w\-]+\.)+)([a-zA-Z]{2,4}))$").unwrap();
        return re.is_match(email.as_ref())
    }
    pub fn email_address_simple<T: AsRef<str>>(email: T) -> bool{
        let re = Regex::new(r"\S+@\S+").unwrap();
        return re.is_match(email.as_ref())
    }
    /// ## Devloper Notes
    /// 
    /// Work on this regex
    pub fn phone_number<T: AsRef<str>>(email: T) -> bool {
        let re = Regex::new(r"^(\+\d{1,2}\s)?\(?\d{3}\)?[\s.-]\d{3}[\s.-]\d{4}$").unwrap();
        return re.is_match(email.as_ref())
    }
    pub fn us_zip_code<T: AsRef<str>>(email: T) -> bool {
        let re = Regex::new(r"^\d{5}$").unwrap();
        return re.is_match(email.as_ref())
    }
}

impl CanaryValidateCryptocurrencyAddressAPI {
    pub fn monero_address<T: AsRef<str>>(input: T) -> bool {
        let re = Regex::new(r"4[0-9AB][123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz]{93}").unwrap();
        return re.is_match(input.as_ref())
    }
    pub fn ethereum_address<T: AsRef<str>>(input: T) -> bool {
        let re = Regex::new(r"/^0x[a-fA-F0-9]{40}$/").unwrap();
        return re.is_match(input.as_ref())
    }
    pub fn bitcoin_address<T: AsRef<str>>(input: T) -> bool {
        let re = Regex::new(r"^[1]([123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz]*)$").unwrap();
        return re.is_match(input.as_ref())
    }
}

impl CanaryValidateEncodingAPI {
    pub fn base32<T: AsRef<str>>(input: T) -> bool{
        let re = Regex::new(r"^(?:[A-Z2-7]{8})*(?:[A-Z2-7]{2}={6}|[A-Z2-7]{4}={4}|[A-Z2-7]{5}={3}|[A-Z2-7]{7}=)?$").unwrap();
        return re.is_match(input.as_ref())
    }
    pub fn base58<T: AsRef<str>>(input: T) -> bool {
        let re = Regex::new(r"^[1]([123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz]*)$").unwrap();
        return re.is_match(input.as_ref())
    }
    pub fn base64<T: AsRef<str>>(input: T) -> bool {
        let re = Regex::new(r"/^(?:[A-Za-z\d+/]{4})*(?:[A-Za-z\d+/]{3}=|[A-Za-z\d+/]{2}==)?$/").unwrap();
        return re.is_match(input.as_ref())
    }
    pub fn hex<T: AsRef<str>>(input: T) -> bool {
        let re = Regex::new(r"^(0x|0X)?[a-fA-F0-9]+$'").unwrap();
        return re.is_match(input.as_ref())        
    }
}

#[test]
fn single_digit(){
    assert!(CanaryValidateDigitAPI::single_digit("0"));
    assert!(CanaryValidateDigitAPI::single_digit("1"));
    assert!(CanaryValidateDigitAPI::single_digit("2"));
    assert!(CanaryValidateDigitAPI::single_digit("3"));
    assert!(CanaryValidateDigitAPI::single_digit("4"));
    assert!(CanaryValidateDigitAPI::single_digit("5"));
    assert!(CanaryValidateDigitAPI::single_digit("6"));
    assert!(CanaryValidateDigitAPI::single_digit("7"));
    assert!(CanaryValidateDigitAPI::single_digit("8"));
    assert!(CanaryValidateDigitAPI::single_digit("9"));
}
