## ----Loading datasets, message=FALSE------------------------------------------
library(fcoex, quietly = TRUE)
library(SingleCellExperiment, quietly = TRUE)
data("mini_pbmc3k")

## ----Creating fcoex object, message=FALSE-------------------------------------
# Get clusters from the pre-processing
target <- colData(mini_pbmc3k)
target <- target$clusters

# Get normalized table from the pre-processing
exprs <- as.data.frame(assay(mini_pbmc3k, 'logcounts'))

# Create fcoex object
fc <- new_fcoex(data.frame(exprs),target)

## ----Discretizing dataset, message=FALSE--------------------------------------
fc <- discretize(fc, number_of_bins = 8)

## ----Finding cbf modules, message=FALSE---------------------------------------
fc <- find_cbf_modules(fc,n_genes_selected_in_first_step = 200, verbose = FALSE, is_parallel = FALSE)

## ----Plotting module networks, message=FALSE----------------------------------
fc <- get_nets(fc)

# Taking a look at the first two networks: 
network_plots <- show_net(fc)

network_plots[["CD79A"]]
network_plots[["HLA-DRB1"]]

## ----Saving plots, eval= FALSE, message=FALSE, results='hide'-----------------
#  save_plots(name = "fcoex_vignette", fc,force = TRUE, directory = "./Plots")

## ----Running ORA analysis, warning=FALSE--------------------------------------

# You'll need CEMiTool, if you do not have it installed, just run BiocManager::install("CEMiTool")
gmt_filename <- system.file("extdata", "pathways.gmt", package = "CEMiTool")

if (gmt_filename == "")
  {
      print("You likely need to install CEMiTool")
} else {
      gmt_in <- pathwayPCA::read_gmt(gmt_filename,  description = TRUE)

}
fc <- mod_ora(fc, gmt_in)
fc <- plot_ora(fc)

## ----Saving plots again,  eval= FALSE, message=FALSE, results='hide'----------
#  save_plots(name = "fcoex_vignette", fc, force = TRUE, directory = "./Plots")

## ----Reclustering , message=FALSE---------------------------------------------
fc <- recluster(fc)

## ----Visualizing--------------------------------------------------------------
identities_based_on_the_HLA_DRB1_module <- idents(fc)$`HLA-DRB1` 
colData(mini_pbmc3k) <- cbind(colData(mini_pbmc3k), `mod_HLA_DRB1` = identities_based_on_the_HLA_DRB1_module )

identities_based_on_the__CD79A_module <- idents(fc)$`HLA-DRB1` 
colData(mini_pbmc3k) <- cbind(colData(mini_pbmc3k), mod_CD79A = idents(fc)$CD79A)

# Let's see the original clusters
library(schex)

mini_pbmc3k <- make_hexbin(mini_pbmc3k, nbins = 40, 
    dimension_reduction = "UMAP", use_dims=c(1,2))

plot_hexbin_meta(mini_pbmc3k, col="clusters", action="majority")

library(gridExtra)
p1 = plot_hexbin_feature_plus(mini_pbmc3k,
    col="clusters", type="logcounts",
    feature="CD79A", action="mean") +
  ggtitle("original clusters (CD79A expression)") +
  theme_void()

p2 =plot_hexbin_feature_plus(mini_pbmc3k,
    col="clusters", type="logcounts",
    feature="HLA-DRB1", action="mean") +
  ggtitle("original clusters (HLA-DRB1 expression)") +
  theme_void()

p3 =  plot_hexbin_feature_plus(mini_pbmc3k,
    col="mod_CD79A", type="logcounts",
    feature="CD79A", action="mean") +
  ggtitle("fcoex CD79A clusters (CD79A expression)") +
  theme_void()

p4 =  plot_hexbin_feature_plus(mini_pbmc3k,
    col="mod_HLA_DRB1", type="logcounts",
    feature="HLA-DRB1", action="mean")+
  ggtitle("fcoex HLA cluster (HLA-DRB1 expression)") +
  theme_void()

grid.arrange(p1, p2, p3, p4, nrow=2)

