## ----global-options, include=FALSE--------------------------------------------
knitr::opts_chunk$set(
  message=FALSE,
  warning=FALSE,
  collapse=TRUE,
  comment="#>"
)
library(BiocStyle)

## ---- eval=FALSE--------------------------------------------------------------
#  if (!require("BiocManager", quietly = TRUE))
#      install.packages("BiocManager")
#  BiocManager::install("MoleculeExperiment")

## -----------------------------------------------------------------------------
library(MoleculeExperiment)
library(ggplot2)

## -----------------------------------------------------------------------------
repoDir <- system.file("extdata", package = "MoleculeExperiment")
repoDir <- paste0(repoDir, "/xenium_V1_FF_Mouse_Brain")

me <- readXenium(repoDir,
                  keepCols = "essential")
me

## -----------------------------------------------------------------------------
ggplot_me() +
      geom_polygon_me(me, assayName = "cell", fill = "grey") +
      geom_point_me(me) +
      # zoom in to selected patch area
      coord_cartesian(xlim = c(4900, 4919.98), 
                        ylim = c(6400.02, 6420))

## -----------------------------------------------------------------------------
# transform ME to SPE object
spe <- countMolecules(me)
spe

## -----------------------------------------------------------------------------
moleculesDf <- data.frame(
  sample_id = rep(c("sample1", "sample2"), times = c(30, 20)),
  features = rep(c("gene1", "gene2"), times = c(20, 30)),
  x_coords = runif(50),
  y_coords = runif(50)
  )
head(moleculesDf)

## -----------------------------------------------------------------------------
boundariesDf <- data.frame(
  sample_id = rep(c("sample1", "sample2"), times = c(16, 6)),
  cell_id = rep(c("cell1", "cell2", "cell3", "cell4",
                  "cell1", "cell2"),
                times = c(4, 4, 4, 4, 3, 3)),
  vertex_x = c(0, 0.5, 0.5, 0, 
               0.5, 1, 1, 0.5, 
               0, 0.5, 0.5, 0, 
               0.5, 1, 1, 0.5,
               0, 1, 0, 0, 1, 1),
  vertex_y = c(0, 0, 0.5, 0.5, 
               0, 0, 0.5, 0.5, 
               0.5, 0.5, 1, 1, 
               0.5, 0.5, 1, 1,
               0, 1, 1, 0, 0, 1)
)
head(boundariesDf)

## -----------------------------------------------------------------------------
moleculesMEList <- dataframeToMEList(moleculesDf,
                                  dfType = "transcripts",
                                  assayName = "detected",
                                  sampleCol = "sample_id",
                                  factorCol = "features",
                                  xCol = "x_coords",
                                  yCol = "y_coords")
str(moleculesMEList, max.level = 3)

## -----------------------------------------------------------------------------
boundariesMEList <- dataframeToMEList(boundariesDf,
                                   dfType = "boundaries",
                                   assayName = "cell",
                                   sampleCol = "sample_id",
                                   factorCol = "cell_id",
                                   xCol = "vertex_x",
                                   yCol = "vertex_y")
str(boundariesMEList, 3)

## -----------------------------------------------------------------------------
toyME <- MoleculeExperiment(molecules = moleculesMEList,
                             boundaries = boundariesMEList)
toyME

## -----------------------------------------------------------------------------
repoDir <- system.file("extdata", package = "MoleculeExperiment")
repoDir <- paste0(repoDir, "/xenium_V1_FF_Mouse_Brain")

me <- readXenium(repoDir,
                  keepCols = "essential",
                  addBoundaries = "cell")
me

## -----------------------------------------------------------------------------
repoDir <- system.file("extdata", package = "MoleculeExperiment")
repoDir <- paste0(repoDir, "/nanostring_Lung9_Rep1")

meCosmx <- readCosmx(repoDir,
                      keepCols = "essential")
meCosmx
# boundaries not handled yet, as CosMx does not provide cell boundary info

## -----------------------------------------------------------------------------
repoDir <- system.file("extdata", package = "MoleculeExperiment")
repoDir <- paste0(repoDir, "/vizgen_HumanOvarianCancerPatient2Slice2")
meMerscope <- readMerscope(repoDir,
                          keepCols = "essential")
meMerscope
# boundaries not handled yet, as Merscope has a complex set of multitude
# boundary files in hdf5 file format

## -----------------------------------------------------------------------------
showMolecules(me)

## -----------------------------------------------------------------------------
showBoundaries(me)

## ---- echo=TRUE, results= 'hide'----------------------------------------------
# NOTE: output not shown as it is too large
# access molecules slot
molecules(me)
# access cell boundary information in boundaries slot
boundaries(me, "cell")

## -----------------------------------------------------------------------------
molecules(me, assayName = "detected", flatten = TRUE)

## -----------------------------------------------------------------------------
boundaries(me, assayName = "cell", flatten = TRUE)

## -----------------------------------------------------------------------------
# get initial features in sample 1
head(features(me)[[1]])

## -----------------------------------------------------------------------------
segmentIDs(me, "cell")

## -----------------------------------------------------------------------------
repoDir <- system.file("extdata", package = "MoleculeExperiment")
repoDir <- paste0(repoDir, "/xenium_V1_FF_Mouse_Brain")
nucleiMEList <- readBoundaries(dataDir = repoDir,
                            pattern = "nucleus_boundaries.csv",
                            segmentIDCol = "cell_id",
                            xCol = "vertex_x",
                            yCol = "vertex_y",
                            keepCols = "essential",
                            boundariesAssay = "nucleus",
                            scaleFactorVector = 1)

boundaries(me, "nucleus") <- nucleiMEList
me # note the addition of the nucleus boundaries to the boundaries slot

## -----------------------------------------------------------------------------
ggplot_me() +
  # add cell segments and colour by cell id
  geom_polygon_me(me, byFill = "segment_id", colour = "black", alpha = 0.1) +
  # add molecule points and colour by feature name
  geom_point_me(me, byColour = "feature_name", size = 0.1) +
  # add nuclei segments and colour the border with red
  geom_polygon_me(me, assayName = "nucleus", fill = NA, colour = "red") +
  # zoom in to selected patch area
  coord_cartesian(xlim = c(4900, 4919.98), ylim = c(6400.02, 6420))

## -----------------------------------------------------------------------------
spe <- countMolecules(me, boundariesAssay = "nucleus")
spe

## -----------------------------------------------------------------------------
sessionInfo()

